# frozen_string_literal: true

#--
# Copyright (C) Bob Aman
#
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.
#++


module Addressable
  module IDNA
    # This module is loosely based on idn_actionmailer by Mick Staugaard,
    # the unicode library by Yoshida Masato, and the punycode implementation
    # by Kazuhiro Nishiyama.  Most of the code was copied verbatim, but
    # some reformatting was done, and some translation from C was done.
    #
    # Without their code to work from as a base, we'd all still be relying
    # on the presence of libidn.  Which nobody ever seems to have installed.
    #
    # Original sources:
    # http://github.com/staugaard/idn_actionmailer
    # http://www.yoshidam.net/Ruby.html#unicode
    # http://rubyforge.org/frs/?group_id=2550

    ACE_PREFIX = "xn--"

    UTF8_REGEX = /\A(?:
      [\x09\x0A\x0D\x20-\x7E]               # ASCII
      | [\xC2-\xDF][\x80-\xBF]              # non-overlong 2-byte
      | \xE0[\xA0-\xBF][\x80-\xBF]          # excluding overlongs
      | [\xE1-\xEC\xEE\xEF][\x80-\xBF]{2}   # straight 3-byte
      | \xED[\x80-\x9F][\x80-\xBF]          # excluding surrogates
      | \xF0[\x90-\xBF][\x80-\xBF]{2}       # planes 1-3
      | [\xF1-\xF3][\x80-\xBF]{3}           # planes 4nil5
      | \xF4[\x80-\x8F][\x80-\xBF]{2}       # plane 16
      )*\z/mnx

    UTF8_REGEX_MULTIBYTE = /(?:
      [\xC2-\xDF][\x80-\xBF]                # non-overlong 2-byte
      | \xE0[\xA0-\xBF][\x80-\xBF]          # excluding overlongs
      | [\xE1-\xEC\xEE\xEF][\x80-\xBF]{2}   # straight 3-byte
      | \xED[\x80-\x9F][\x80-\xBF]          # excluding surrogates
      | \xF0[\x90-\xBF][\x80-\xBF]{2}       # planes 1-3
      | [\xF1-\xF3][\x80-\xBF]{3}           # planes 4nil5
      | \xF4[\x80-\x8F][\x80-\xBF]{2}       # plane 16
      )/mnx

    # :startdoc:

    # Converts from a Unicode internationalized domain name to an ASCII
    # domain name as described in RFC 3490.
    def self.to_ascii(input)
      input = input.to_s unless input.is_a?(String)
      input = input.dup.force_encoding(Encoding::UTF_8).unicode_normalize(:nfkc)
      if input.respond_to?(:force_encoding)
        input.force_encoding(Encoding::ASCII_8BIT)
      end
      if input =~ UTF8_REGEX && input =~ UTF8_REGEX_MULTIBYTE
        parts = unicode_downcase(input).split('.')
        parts.map! do |part|
          if part.respond_to?(:force_encoding)
            part.force_encoding(Encoding::ASCII_8BIT)
          end
          if part =~ UTF8_REGEX && part =~ UTF8_REGEX_MULTIBYTE
            ACE_PREFIX + punycode_encode(part)
          else
            part
          end
        end
        parts.join('.')
      else
        input
      end
    end

    # Converts from an ASCII domain name to a Unicode internationalized
    # domain name as described in RFC 3490.
    def self.to_unicode(input)
      input = input.to_s unless input.is_a?(String)
      parts = input.split('.')
      parts.map! do |part|
        if part =~ /^#{ACE_PREFIX}(.+)/
          begin
            punycode_decode(part[/^#{ACE_PREFIX}(.+)/, 1])
          rescue Addressable::IDNA::PunycodeBadInput
            # toUnicode is explicitly defined as never-fails by the spec
            part
          end
        else
          part
        end
      end
      output = parts.join('.')
      if output.respond_to?(:force_encoding)
        output.force_encoding(Encoding::UTF_8)
      end
      output
    end

    class << self
      # @deprecated Use {String#unicode_normalize(:nfkc)} instead
      def unicode_normalize_kc(value)
        value.to_s.unicode_normalize(:nfkc)
      end

      extend Gem::Deprecate
      deprecate :unicode_normalize_kc, "String#unicode_normalize(:nfkc)", 2023, 4
    end

    ##
    # Unicode aware downcase method.
    #
    # @api private
    # @param [String] input
    #   The input string.
    # @return [String] The downcased result.
    def self.unicode_downcase(input)
      input = input.to_s unless input.is_a?(String)
      unpacked = input.unpack("U*")
      unpacked.map! { |codepoint| lookup_unicode_lowercase(codepoint) }
      return unpacked.pack("U*")
    end
    private_class_method :unicode_downcase

    def self.lookup_unicode_lowercase(codepoint)
      codepoint_data = UNICODE_DATA[codepoint]
      (codepoint_data ?
        (codepoint_data[UNICODE_DATA_LOWERCASE] || codepoint) :
        codepoint)
    end
    private_class_method :lookup_unicode_lowercase

    UNICODE_DATA_COMBINING_CLASS = 0
    UNICODE_DATA_EXCLUSION = 1
    UNICODE_DATA_CANONICAL = 2
    UNICODE_DATA_COMPATIBILITY = 3
    UNICODE_DATA_UPPERCASE = 4
    UNICODE_DATA_LOWERCASE = 5
    UNICODE_DATA_TITLECASE = 6

    UNICODE_DATA = {
      65 => [0, 0, nil, nil, nil, 97, nil],
      66 => [0, 0, nil, nil, nil, 98, nil],
      67 => [0, 0, nil, nil, nil, 99, nil],
      68 => [0, 0, nil, nil, nil, 100, nil],
      69 => [0, 0, nil, nil, nil, 101, nil],
      70 => [0, 0, nil, nil, nil, 102, nil],
      71 => [0, 0, nil, nil, nil, 103, nil],
      72 => [0, 0, nil, nil, nil, 104, nil],
      73 => [0, 0, nil, nil, nil, 105, nil],
      74 => [0, 0, nil, nil, nil, 106, nil],
      75 => [0, 0, nil, nil, nil, 107, nil],
      76 => [0, 0, nil, nil, nil, 108, nil],
      77 => [0, 0, nil, nil, nil, 109, nil],
      78 => [0, 0, nil, nil, nil, 110, nil],
      79 => [0, 0, nil, nil, nil, 111, nil],
      80 => [0, 0, nil, nil, nil, 112, nil],
      81 => [0, 0, nil, nil, nil, 113, nil],
      82 => [0, 0, nil, nil, nil, 114, nil],
      83 => [0, 0, nil, nil, nil, 115, nil],
      84 => [0, 0, nil, nil, nil, 116, nil],
      85 => [0, 0, nil, nil, nil, 117, nil],
      86 => [0, 0, nil, nil, nil, 118, nil],
      87 => [0, 0, nil, nil, nil, 119, nil],
      88 => [0, 0, nil, nil, nil, 120, nil],
      89 => [0, 0, nil, nil, nil, 121, nil],
      90 => [0, 0, nil, nil, nil, 122, nil],
      97 => [0, 0, nil, nil, 65, nil, 65],
      98 => [0, 0, nil, nil, 66, nil, 66],
      99 => [0, 0, nil, nil, 67, nil, 67],
      100 => [0, 0, nil, nil, 68, nil, 68],
      101 => [0, 0, nil, nil, 69, nil, 69],
      102 => [0, 0, nil, nil, 70, nil, 70],
      103 => [0, 0, nil, nil, 71, nil, 71],
      104 => [0, 0, nil, nil, 72, nil, 72],
      105 => [0, 0, nil, nil, 73, nil, 73],
      106 => [0, 0, nil, nil, 74, nil, 74],
      107 => [0, 0, nil, nil, 75, nil, 75],
      108 => [0, 0, nil, nil, 76, nil, 76],
      109 => [0, 0, nil, nil, 77, nil, 77],
      110 => [0, 0, nil, nil, 78, nil, 78],
      111 => [0, 0, nil, nil, 79, nil, 79],
      112 => [0, 0, nil, nil, 80, nil, 80],
      113 => [0, 0, nil, nil, 81, nil, 81],
      114 => [0, 0, nil, nil, 82, nil, 82],
      115 => [0, 0, nil, nil, 83, nil, 83],
      116 => [0, 0, nil, nil, 84, nil, 84],
      117 => [0, 0, nil, nil, 85, nil, 85],
      118 => [0, 0, nil, nil, 86, nil, 86],
      119 => [0, 0, nil, nil, 87, nil, 87],
      120 => [0, 0, nil, nil, 88, nil, 88],
      121 => [0, 0, nil, nil, 89, nil, 89],
      122 => [0, 0, nil, nil, 90, nil, 90],
      160 => [0, 0, nil, " ", nil, nil, nil],
      168 => [0, 0, nil, " ̈", nil, nil, nil],
      170 => [0, 0, nil, "a", nil, nil, nil],
      175 => [0, 0, nil, " ̄", nil, nil, nil],
      178 => [0, 0, nil, "2", nil, nil, nil],
      179 => [0, 0, nil, "3", nil, nil, nil],
      180 => [0, 0, nil, " ́", nil, nil, nil],
      181 => [0, 0, nil, "μ", 924, nil, 924],
      184 => [0, 0, nil, " ̧", nil, nil, nil],
      185 => [0, 0, nil, "1", nil, nil, nil],
      186 => [0, 0, nil, "o", nil, nil, nil],
      188 => [0, 0, nil, "1⁄4", nil, nil, nil],
      189 => [0, 0, nil, "1⁄2", nil, nil, nil],
      190 => [0, 0, nil, "3⁄4", nil, nil, nil],
      192 => [0, 0, "À", "À", nil, 224, nil],
      193 => [0, 0, "Á", "Á", nil, 225, nil],
      194 => [0, 0, "Â", "Â", nil, 226, nil],
      195 => [0, 0, "Ã", "Ã", nil, 227, nil],
      196 => [0, 0, "Ä", "Ä", nil, 228, nil],
      197 => [0, 0, "Å", "Å", nil, 229, nil],
      198 => [0, 0, nil, nil, nil, 230, nil],
      199 => [0, 0, "Ç", "Ç", nil, 231, nil],
      200 => [0, 0, "È", "È", nil, 232, nil],
      201 => [0, 0, "É", "É", nil, 233, nil],
      202 => [0, 0, "Ê", "Ê", nil, 234, nil],
      203 => [0, 0, "Ë", "Ë", nil, 235, nil],
      204 => [0, 0, "Ì", "Ì", nil, 236, nil],
      205 => [0, 0, "Í", "Í", nil, 237, nil],
      206 => [0, 0, "Î", "Î", nil, 238, nil],
      207 => [0, 0, "Ï", "Ï", nil, 239, nil],
      208 => [0, 0, nil, nil, nil, 240, nil],
      209 => [0, 0, "Ñ", "Ñ", nil, 241, nil],
      210 => [0, 0, "Ò", "Ò", nil, 242, nil],
      211 => [0, 0, "Ó", "Ó", nil, 243, nil],
      212 => [0, 0, "Ô", "Ô", nil, 244, nil],
      213 => [0, 0, "Õ", "Õ", nil, 245, nil],
      214 => [0, 0, "Ö", "Ö", nil, 246, nil],
      216 => [0, 0, nil, nil, nil, 248, nil],
      217 => [0, 0, "Ù", "Ù", nil, 249, nil],
      218 => [0, 0, "Ú", "Ú", nil, 250, nil],
      219 => [0, 0, "Û", "Û", nil, 251, nil],
      220 => [0, 0, "Ü", "Ü", nil, 252, nil],
      221 => [0, 0, "Ý", "Ý", nil, 253, nil],
      222 => [0, 0, nil, nil, nil, 254, nil],
      224 => [0, 0, "à", "à", 192, nil, 192],
      225 => [0, 0, "á", "á", 193, nil, 193],
      226 => [0, 0, "â", "â", 194, nil, 194],
      227 => [0, 0, "ã", "ã", 195, nil, 195],
      228 => [0, 0, "ä", "ä", 196, nil, 196],
      229 => [0, 0, "å", "å", 197, nil, 197],
      230 => [0, 0, nil, nil, 198, nil, 198],
      231 => [0, 0, "ç", "ç", 199, nil, 199],
      232 => [0, 0, "è", "è", 200, nil, 200],
      233 => [0, 0, "é", "é", 201, nil, 201],
      234 => [0, 0, "ê", "ê", 202, nil, 202],
      235 => [0, 0, "ë", "ë", 203, nil, 203],
      236 => [0, 0, "ì", "ì", 204, nil, 204],
      237 => [0, 0, "í", "í", 205, nil, 205],
      238 => [0, 0, "î", "î", 206, nil, 206],
      239 => [0, 0, "ï", "ï", 207, nil, 207],
      240 => [0, 0, nil, nil, 208, nil, 208],
      241 => [0, 0, "ñ", "ñ", 209, nil, 209],
      242 => [0, 0, "ò", "ò", 210, nil, 210],
      243 => [0, 0, "ó", "ó", 211, nil, 211],
      244 => [0, 0, "ô", "ô", 212, nil, 212],
      245 => [0, 0, "õ", "õ", 213, nil, 213],
      246 => [0, 0, "ö", "ö", 214, nil, 214],
      248 => [0, 0, nil, nil, 216, nil, 216],
      249 => [0, 0, "ù", "ù", 217, nil, 217],
      250 => [0, 0, "ú", "ú", 218, nil, 218],
      251 => [0, 0, "û", "û", 219, nil, 219],
      252 => [0, 0, "ü", "ü", 220, nil, 220],
      253 => [0, 0, "ý", "ý", 221, nil, 221],
      254 => [0, 0, nil, nil, 222, nil, 222],
      255 => [0, 0, "ÿ", "ÿ", 376, nil, 376],
      256 => [0, 0, "Ā", "Ā", nil, 257, nil],
      257 => [0, 0, "ā", "ā", 256, nil, 256],
      258 => [0, 0, "Ă", "Ă", nil, 259, nil],
      259 => [0, 0, "ă", "ă", 258, nil, 258],
      260 => [0, 0, "Ą", "Ą", nil, 261, nil],
      261 => [0, 0, "ą", "ą", 260, nil, 260],
      262 => [0, 0, "Ć", "Ć", nil, 263, nil],
      263 => [0, 0, "ć", "ć", 262, nil, 262],
      264 => [0, 0, "Ĉ", "Ĉ", nil, 265, nil],
      265 => [0, 0, "ĉ", "ĉ", 264, nil, 264],
      266 => [0, 0, "Ċ", "Ċ", nil, 267, nil],
      267 => [0, 0, "ċ", "ċ", 266, nil, 266],
      268 => [0, 0, "Č", "Č", nil, 269, nil],
      269 => [0, 0, "č", "č", 268, nil, 268],
      270 => [0, 0, "Ď", "Ď", nil, 271, nil],
      271 => [0, 0, "ď", "ď", 270, nil, 270],
      272 => [0, 0, nil, nil, nil, 273, nil],
      273 => [0, 0, nil, nil, 272, nil, 272],
      274 => [0, 0, "Ē", "Ē", nil, 275, nil],
      275 => [0, 0, "ē", "ē", 274, nil, 274],
      276 => [0, 0, "Ĕ", "Ĕ", nil, 277, nil],
      277 => [0, 0, "ĕ", "ĕ", 276, nil, 276],
      278 => [0, 0, "Ė", "Ė", nil, 279, nil],
      279 => [0, 0, "ė", "ė", 278, nil, 278],
      280 => [0, 0, "Ę", "Ę", nil, 281, nil],
      281 => [0, 0, "ę", "ę", 280, nil, 280],
      282 => [0, 0, "Ě", "Ě", nil, 283, nil],
      283 => [0, 0, "ě", "ě", 282, nil, 282],
      284 => [0, 0, "Ĝ", "Ĝ", nil, 285, nil],
      285 => [0, 0, "ĝ", "ĝ", 284, nil, 284],
      286 => [0, 0, "Ğ", "Ğ", nil, 287, nil],
      287 => [0, 0, "ğ", "ğ", 286, nil, 286],
      288 => [0, 0, "Ġ", "Ġ", nil, 289, nil],
      289 => [0, 0, "ġ", "ġ", 288, nil, 288],
      290 => [0, 0, "Ģ", "Ģ", nil, 291, nil],
      291 => [0, 0, "ģ", "ģ", 290, nil, 290],
      292 => [0, 0, "Ĥ", "Ĥ", nil, 293, nil],
      293 => [0, 0, "ĥ", "ĥ", 292, nil, 292],
      294 => [0, 0, nil, nil, nil, 295, nil],
      295 => [0, 0, nil, nil, 294, nil, 294],
      296 => [0, 0, "Ĩ", "Ĩ", nil, 297, nil],
      297 => [0, 0, "ĩ", "ĩ", 296, nil, 296],
      298 => [0, 0, "Ī", "Ī", nil, 299, nil],
      299 => [0, 0, "ī", "ī", 298, nil, 298],
      300 => [0, 0, "Ĭ", "Ĭ", nil, 301, nil],
      301 => [0, 0, "ĭ", "ĭ", 300, nil, 300],
      302 => [0, 0, "Į", "Į", nil, 303, nil],
      303 => [0, 0, "į", "į", 302, nil, 302],
      304 => [0, 0, "İ", "İ", nil, 105, nil],
      305 => [0, 0, nil, nil, 73, nil, 73],
      306 => [0, 0, nil, "IJ", nil, 307, nil],
      307 => [0, 0, nil, "ij", 306, nil, 306],
      308 => [0, 0, "Ĵ", "Ĵ", nil, 309, nil],
      309 => [0, 0, "ĵ", "ĵ", 308, nil, 308],
      310 => [0, 0, "Ķ", "Ķ", nil, 311, nil],
      311 => [0, 0, "ķ", "ķ", 310, nil, 310],
      313 => [0, 0, "Ĺ", "Ĺ", nil, 314, nil],
      314 => [0, 0, "ĺ", "ĺ", 313, nil, 313],
      315 => [0, 0, "Ļ", "Ļ", nil, 316, nil],
      316 => [0, 0, "ļ", "ļ", 315, nil, 315],
      317 => [0, 0, "Ľ", "Ľ", nil, 318, nil],
      318 => [0, 0, "ľ", "ľ", 317, nil, 317],
      319 => [0, 0, nil, "L·", nil, 320, nil],
      320 => [0, 0, nil, "l·", 319, nil, 319],
      321 => [0, 0, nil, nil, nil, 322, nil],
      322 => [0, 0, nil, nil, 321, nil, 321],
      323 => [0, 0, "Ń", "Ń", nil, 324, nil],
      324 => [0, 0, "ń", "ń", 323, nil, 323],
      325 => [0, 0, "Ņ", "Ņ", nil, 326, nil],
      326 => [0, 0, "ņ", "ņ", 325, nil, 325],
      327 => [0, 0, "Ň", "Ň", nil, 328, nil],
      328 => [0, 0, "ň", "ň", 327, nil, 327],
      329 => [0, 0, nil, "ʼn", nil, nil, nil],
      330 => [0, 0, nil, nil, nil, 331, nil],
      331 => [0, 0, nil, nil, 330, nil, 330],
      332 => [0, 0, "Ō", "Ō", nil, 333, nil],
      333 => [0, 0, "ō", "ō", 332, nil, 332],
      334 => [0, 0, "Ŏ", "Ŏ", nil, 335, nil],
      335 => [0, 0, "ŏ", "ŏ", 334, nil, 334],
      336 => [0, 0, "Ő", "Ő", nil, 337, nil],
      337 => [0, 0, "ő", "ő", 336, nil, 336],
      338 => [0, 0, nil, nil, nil, 339, nil],
      339 => [0, 0, nil, nil, 338, nil, 338],
      340 => [0, 0, "Ŕ", "Ŕ", nil, 341, nil],
      341 => [0, 0, "ŕ", "ŕ", 340, nil, 340],
      342 => [0, 0, "Ŗ", "Ŗ", nil, 343, nil],
      343 => [0, 0, "ŗ", "ŗ", 342, nil, 342],
      344 => [0, 0, "Ř", "Ř", nil, 345, nil],
      345 => [0, 0, "ř", "ř", 344, nil, 344],
      346 => [0, 0, "Ś", "Ś", nil, 347, nil],
      347 => [0, 0, "ś", "ś", 346, nil, 346],
      348 => [0, 0, "Ŝ", "Ŝ", nil, 349, nil],
      349 => [0, 0, "ŝ", "ŝ", 348, nil, 348],
      350 => [0, 0, "Ş", "Ş", nil, 351, nil],
      351 => [0, 0, "ş", "ş", 350, nil, 350],
      352 => [0, 0, "Š", "Š", nil, 353, nil],
      353 => [0, 0, "š", "š", 352, nil, 352],
      354 => [0, 0, "Ţ", "Ţ", nil, 355, nil],
      355 => [0, 0, "ţ", "ţ", 354, nil, 354],
      356 => [0, 0, "Ť", "Ť", nil, 357, nil],
      357 => [0, 0, "ť", "ť", 356, nil, 356],
      358 => [0, 0, nil, nil, nil, 359, nil],
      359 => [0, 0, nil, nil, 358, nil, 358],
      360 => [0, 0, "Ũ", "Ũ", nil, 361, nil],
      361 => [0, 0, "ũ", "ũ", 360, nil, 360],
      362 => [0, 0, "Ū", "Ū", nil, 363, nil],
      363 => [0, 0, "ū", "ū", 362, nil, 362],
      364 => [0, 0, "Ŭ", "Ŭ", nil, 365, nil],
      365 => [0, 0, "ŭ", "ŭ", 364, nil, 364],
      366 => [0, 0, "Ů", "Ů", nil, 367, nil],
      367 => [0, 0, "ů", "ů", 366, nil, 366],
      368 => [0, 0, "Ű", "Ű", nil, 369, nil],
      369 => [0, 0, "ű", "ű", 368, nil, 368],
      370 => [0, 0, "Ų", "Ų", nil, 371, nil],
      371 => [0, 0, "ų", "ų", 370, nil, 370],
      372 => [0, 0, "Ŵ", "Ŵ", nil, 373, nil],
      373 => [0, 0, "ŵ", "ŵ", 372, nil, 372],
      374 => [0, 0, "Ŷ", "Ŷ", nil, 375, nil],
      375 => [0, 0, "ŷ", "ŷ", 374, nil, 374],
      376 => [0, 0, "Ÿ", "Ÿ", nil, 255, nil],
      377 => [0, 0, "Ź", "Ź", nil, 378, nil],
      378 => [0, 0, "ź", "ź", 377, nil, 377],
      379 => [0, 0, "Ż", "Ż", nil, 380, nil],
      380 => [0, 0, "ż", "ż", 379, nil, 379],
      381 => [0, 0, "Ž", "Ž", nil, 382, nil],
      382 => [0, 0, "ž", "ž", 381, nil, 381],
      383 => [0, 0, nil, "s", 83, nil, 83],
      385 => [0, 0, nil, nil, nil, 595, nil],
      386 => [0, 0, nil, nil, nil, 387, nil],
      387 => [0, 0, nil, nil, 386, nil, 386],
      388 => [0, 0, nil, nil, nil, 389, nil],
      389 => [0, 0, nil, nil, 388, nil, 388],
      390 => [0, 0, nil, nil, nil, 596, nil],
      391 => [0, 0, nil, nil, nil, 392, nil],
      392 => [0, 0, nil, nil, 391, nil, 391],
      393 => [0, 0, nil, nil, nil, 598, nil],
      394 => [0, 0, nil, nil, nil, 599, nil],
      395 => [0, 0, nil, nil, nil, 396, nil],
      396 => [0, 0, nil, nil, 395, nil, 395],
      398 => [0, 0, nil, nil, nil, 477, nil],
      399 => [0, 0, nil, nil, nil, 601, nil],
      400 => [0, 0, nil, nil, nil, 603, nil],
      401 => [0, 0, nil, nil, nil, 402, nil],
      402 => [0, 0, nil, nil, 401, nil, 401],
      403 => [0, 0, nil, nil, nil, 608, nil],
      404 => [0, 0, nil, nil, nil, 611, nil],
      405 => [0, 0, nil, nil, 502, nil, 502],
      406 => [0, 0, nil, nil, nil, 617, nil],
      407 => [0, 0, nil, nil, nil, 616, nil],
      408 => [0, 0, nil, nil, nil, 409, nil],
      409 => [0, 0, nil, nil, 408, nil, 408],
      412 => [0, 0, nil, nil, nil, 623, nil],
      413 => [0, 0, nil, nil, nil, 626, nil],
      415 => [0, 0, nil, nil, nil, 629, nil],
      416 => [0, 0, "Ơ", "Ơ", nil, 417, nil],
      417 => [0, 0, "ơ", "ơ", 416, nil, 416],
      418 => [0, 0, nil, nil, nil, 419, nil],
      419 => [0, 0, nil, nil, 418, nil, 418],
      420 => [0, 0, nil, nil, nil, 421, nil],
      421 => [0, 0, nil, nil, 420, nil, 420],
      422 => [0, 0, nil, nil, nil, 640, nil],
      423 => [0, 0, nil, nil, nil, 424, nil],
      424 => [0, 0, nil, nil, 423, nil, 423],
      425 => [0, 0, nil, nil, nil, 643, nil],
      428 => [0, 0, nil, nil, nil, 429, nil],
      429 => [0, 0, nil, nil, 428, nil, 428],
      430 => [0, 0, nil, nil, nil, 648, nil],
      431 => [0, 0, "Ư", "Ư", nil, 432, nil],
      432 => [0, 0, "ư", "ư", 431, nil, 431],
      433 => [0, 0, nil, nil, nil, 650, nil],
      434 => [0, 0, nil, nil, nil, 651, nil],
      435 => [0, 0, nil, nil, nil, 436, nil],
      436 => [0, 0, nil, nil, 435, nil, 435],
      437 => [0, 0, nil, nil, nil, 438, nil],
      438 => [0, 0, nil, nil, 437, nil, 437],
      439 => [0, 0, nil, nil, nil, 658, nil],
      440 => [0, 0, nil, nil, nil, 441, nil],
      441 => [0, 0, nil, nil, 440, nil, 440],
      444 => [0, 0, nil, nil, nil, 445, nil],
      445 => [0, 0, nil, nil, 444, nil, 444],
      447 => [0, 0, nil, nil, 503, nil, 503],
      452 => [0, 0, nil, "DŽ", nil, 454, 453],
      453 => [0, 0, nil, "Dž", 452, 454, nil],
      454 => [0, 0, nil, "dž", 452, nil, 453],
      455 => [0, 0, nil, "LJ", nil, 457, 456],
      456 => [0, 0, nil, "Lj", 455, 457, nil],
      457 => [0, 0, nil, "lj", 455, nil, 456],
      458 => [0, 0, nil, "NJ", nil, 460, 459],
      459 => [0, 0, nil, "Nj", 458, 460, nil],
      460 => [0, 0, nil, "nj", 458, nil, 459],
      461 => [0, 0, "Ǎ", "Ǎ", nil, 462, nil],
      462 => [0, 0, "ǎ", "ǎ", 461, nil, 461],
      463 => [0, 0, "Ǐ", "Ǐ", nil, 464, nil],
      464 => [0, 0, "ǐ", "ǐ", 463, nil, 463],
      465 => [0, 0, "Ǒ", "Ǒ", nil, 466, nil],
      466 => [0, 0, "ǒ", "ǒ", 465, nil, 465],
      467 => [0, 0, "Ǔ", "Ǔ", nil, 468, nil],
      468 => [0, 0, "ǔ", "ǔ", 467, nil, 467],
      469 => [0, 0, "Ǖ", "Ǖ", nil, 470, nil],
      470 => [0, 0, "ǖ", "ǖ", 469, nil, 469],
      471 => [0, 0, "Ǘ", "Ǘ", nil, 472, nil],
      472 => [0, 0, "ǘ", "ǘ", 471, nil, 471],
      473 => [0, 0, "Ǚ", "Ǚ", nil, 474, nil],
      474 => [0, 0, "ǚ", "ǚ", 473, nil, 473],
      475 => [0, 0, "Ǜ", "Ǜ", nil, 476, nil],
      476 => [0, 0, "ǜ", "ǜ", 475, nil, 475],
      477 => [0, 0, nil, nil, 398, nil, 398],
      478 => [0, 0, "Ǟ", "Ǟ", nil, 479, nil],
      479 => [0, 0, "ǟ", "ǟ", 478, nil, 478],
      480 => [0, 0, "Ǡ", "Ǡ", nil, 481, nil],
      481 => [0, 0, "ǡ", "ǡ", 480, nil, 480],
      482 => [0, 0, "Ǣ", "Ǣ", nil, 483, nil],
      483 => [0, 0, "ǣ", "ǣ", 482, nil, 482],
      484 => [0, 0, nil, nil, nil, 485, nil],
      485 => [0, 0, nil, nil, 484, nil, 484],
      486 => [0, 0, "Ǧ", "Ǧ", nil, 487, nil],
      487 => [0, 0, "ǧ", "ǧ", 486, nil, 486],
      488 => [0, 0, "Ǩ", "Ǩ", nil, 489, nil],
      489 => [0, 0, "ǩ", "ǩ", 488, nil, 488],
      490 => [0, 0, "Ǫ", "Ǫ", nil, 491, nil],
      491 => [0, 0, "ǫ", "ǫ", 490, nil, 490],
      492 => [0, 0, "Ǭ", "Ǭ", nil, 493, nil],
      493 => [0, 0, "ǭ", "ǭ", 492, nil, 492],
      494 => [0, 0, "Ǯ", "Ǯ", nil, 495, nil],
      495 => [0, 0, "ǯ", "ǯ", 494, nil, 494],
      496 => [0, 0, "ǰ", "ǰ", nil, nil, nil],
      497 => [0, 0, nil, "DZ", nil, 499, 498],
      498 => [0, 0, nil, "Dz", 497, 499, nil],
      499 => [0, 0, nil, "dz", 497, nil, 498],
      500 => [0, 0, "Ǵ", "Ǵ", nil, 501, nil],
      501 => [0, 0, "ǵ", "ǵ", 500, nil, 500],
      502 => [0, 0, nil, nil, nil, 405, nil],
      503 => [0, 0, nil, nil, nil, 447, nil],
      504 => [0, 0, "Ǹ", "Ǹ", nil, 505, nil],
      505 => [0, 0, "ǹ", "ǹ", 504, nil, 504],
      506 => [0, 0, "Ǻ", "Ǻ", nil, 507, nil],
      507 => [0, 0, "ǻ", "ǻ", 506, nil, 506],
      508 => [0, 0, "Ǽ", "Ǽ", nil, 509, nil],
      509 => [0, 0, "ǽ", "ǽ", 508, nil, 508],
      510 => [0, 0, "Ǿ", "Ǿ", nil, 511, nil],
      511 => [0, 0, "ǿ", "ǿ", 510, nil, 510],
      512 => [0, 0, "Ȁ", "Ȁ", nil, 513, nil],
      513 => [0, 0, "ȁ", "ȁ", 512, nil, 512],
      514 => [0, 0, "Ȃ", "Ȃ", nil, 515, nil],
      515 => [0, 0, "ȃ", "ȃ", 514, nil, 514],
      516 => [0, 0, "Ȅ", "Ȅ", nil, 517, nil],
      517 => [0, 0, "ȅ", "ȅ", 516, nil, 516],
      518 => [0, 0, "Ȇ", "Ȇ", nil, 519, nil],
      519 => [0, 0, "ȇ", "ȇ", 518, nil, 518],
      520 => [0, 0, "Ȉ", "Ȉ", nil, 521, nil],
      521 => [0, 0, "ȉ", "ȉ", 520, nil, 520],
      522 => [0, 0, "Ȋ", "Ȋ", nil, 523, nil],
      523 => [0, 0, "ȋ", "ȋ", 522, nil, 522],
      524 => [0, 0, "Ȍ", "Ȍ", nil, 525, nil],
      525 => [0, 0, "ȍ", "ȍ", 524, nil, 524],
      526 => [0, 0, "Ȏ", "Ȏ", nil, 527, nil],
      527 => [0, 0, "ȏ", "ȏ", 526, nil, 526],
      528 => [0, 0, "Ȑ", "Ȑ", nil, 529, nil],
      529 => [0, 0, "ȑ", "ȑ", 528, nil, 528],
      530 => [0, 0, "Ȓ", "Ȓ", nil, 531, nil],
      531 => [0, 0, "ȓ", "ȓ", 530, nil, 530],
      532 => [0, 0, "Ȕ", "Ȕ", nil, 533, nil],
      533 => [0, 0, "ȕ", "ȕ", 532, nil, 532],
      534 => [0, 0, "Ȗ", "Ȗ", nil, 535, nil],
      535 => [0, 0, "ȗ", "ȗ", 534, nil, 534],
      536 => [0, 0, "Ș", "Ș", nil, 537, nil],
      537 => [0, 0, "ș", "ș", 536, nil, 536],
      538 => [0, 0, "Ț", "Ț", nil, 539, nil],
      539 => [0, 0, "ț", "ț", 538, nil, 538],
      540 => [0, 0, nil, nil, nil, 541, nil],
      541 => [0, 0, nil, nil, 540, nil, 540],
      542 => [0, 0, "Ȟ", "Ȟ", nil, 543, nil],
      543 => [0, 0, "ȟ", "ȟ", 542, nil, 542],
      546 => [0, 0, nil, nil, nil, 547, nil],
      547 => [0, 0, nil, nil, 546, nil, 546],
      548 => [0, 0, nil, nil, nil, 549, nil],
      549 => [0, 0, nil, nil, 548, nil, 548],
      550 => [0, 0, "Ȧ", "Ȧ", nil, 551, nil],
      551 => [0, 0, "ȧ", "ȧ", 550, nil, 550],
      552 => [0, 0, "Ȩ", "Ȩ", nil, 553, nil],
      553 => [0, 0, "ȩ", "ȩ", 552, nil, 552],
      554 => [0, 0, "Ȫ", "Ȫ", nil, 555, nil],
      555 => [0, 0, "ȫ", "ȫ", 554, nil, 554],
      556 => [0, 0, "Ȭ", "Ȭ", nil, 557, nil],
      557 => [0, 0, "ȭ", "ȭ", 556, nil, 556],
      558 => [0, 0, "Ȯ", "Ȯ", nil, 559, nil],
      559 => [0, 0, "ȯ", "ȯ", 558, nil, 558],
      560 => [0, 0, "Ȱ", "Ȱ", nil, 561, nil],
      561 => [0, 0, "ȱ", "ȱ", 560, nil, 560],
      562 => [0, 0, "Ȳ", "Ȳ", nil, 563, nil],
      563 => [0, 0, "ȳ", "ȳ", 562, nil, 562],
      595 => [0, 0, nil, nil, 385, nil, 385],
      596 => [0, 0, nil, nil, 390, nil, 390],
      598 => [0, 0, nil, nil, 393, nil, 393],
      599 => [0, 0, nil, nil, 394, nil, 394],
      601 => [0, 0, nil, nil, 399, nil, 399],
      603 => [0, 0, nil, nil, 400, nil, 400],
      608 => [0, 0, nil, nil, 403, nil, 403],
      611 => [0, 0, nil, nil, 404, nil, 404],
      616 => [0, 0, nil, nil, 407, nil, 407],
      617 => [0, 0, nil, nil, 406, nil, 406],
      623 => [0, 0, nil, nil, 412, nil, 412],
      626 => [0, 0, nil, nil, 413, nil, 413],
      629 => [0, 0, nil, nil, 415, nil, 415],
      640 => [0, 0, nil, nil, 422, nil, 422],
      643 => [0, 0, nil, nil, 425, nil, 425],
      648 => [0, 0, nil, nil, 430, nil, 430],
      650 => [0, 0, nil, nil, 433, nil, 433],
      651 => [0, 0, nil, nil, 434, nil, 434],
      658 => [0, 0, nil, nil, 439, nil, 439],
      688 => [0, 0, nil, "h", nil, nil, nil],
      689 => [0, 0, nil, "ɦ", nil, nil, nil],
      690 => [0, 0, nil, "j", nil, nil, nil],
      691 => [0, 0, nil, "r", nil, nil, nil],
      692 => [0, 0, nil, "ɹ", nil, nil, nil],
      693 => [0, 0, nil, "ɻ", nil, nil, nil],
      694 => [0, 0, nil, "ʁ", nil, nil, nil],
      695 => [0, 0, nil, "w", nil, nil, nil],
      696 => [0, 0, nil, "y", nil, nil, nil],
      728 => [0, 0, nil, " ̆", nil, nil, nil],
      729 => [0, 0, nil, " ̇", nil, nil, nil],
      730 => [0, 0, nil, " ̊", nil, nil, nil],
      731 => [0, 0, nil, " ̨", nil, nil, nil],
      732 => [0, 0, nil, " ̃", nil, nil, nil],
      733 => [0, 0, nil, " ̋", nil, nil, nil],
      736 => [0, 0, nil, "ɣ", nil, nil, nil],
      737 => [0, 0, nil, "l", nil, nil, nil],
      738 => [0, 0, nil, "s", nil, nil, nil],
      739 => [0, 0, nil, "x", nil, nil, nil],
      740 => [0, 0, nil, "ʕ", nil, nil, nil],
      768 => [230, 0, nil, nil, nil, nil, nil],
      769 => [230, 0, nil, nil, nil, nil, nil],
      770 => [230, 0, nil, nil, nil, nil, nil],
      771 => [230, 0, nil, nil, nil, nil, nil],
      772 => [230, 0, nil, nil, nil, nil, nil],
      773 => [230, 0, nil, nil, nil, nil, nil],
      774 => [230, 0, nil, nil, nil, nil, nil],
      775 => [230, 0, nil, nil, nil, nil, nil],
      776 => [230, 0, nil, nil, nil, nil, nil],
      777 => [230, 0, nil, nil, nil, nil, nil],
      778 => [230, 0, nil, nil, nil, nil, nil],
      779 => [230, 0, nil, nil, nil, nil, nil],
      780 => [230, 0, nil, nil, nil, nil, nil],
      781 => [230, 0, nil, nil, nil, nil, nil],
      782 => [230, 0, nil, nil, nil, nil, nil],
      783 => [230, 0, nil, nil, nil, nil, nil],
      784 => [230, 0, nil, nil, nil, nil, nil],
      785 => [230, 0, nil, nil, nil, nil, nil],
      786 => [230, 0, nil, nil, nil, nil, nil],
      787 => [230, 0, nil, nil, nil, nil, nil],
      788 => [230, 0, nil, nil, nil, nil, nil],
      789 => [232, 0, nil, nil, nil, nil, nil],
      790 => [220, 0, nil, nil, nil, nil, nil],
      791 => [220, 0, nil, nil, nil, nil, nil],
      792 => [220, 0, nil, nil, nil, nil, nil],
      793 => [220, 0, nil, nil, nil, nil, nil],
      794 => [232, 0, nil, nil, nil, nil, nil],
      795 => [216, 0, nil, nil, nil, nil, nil],
      796 => [220, 0, nil, nil, nil, nil, nil],
      797 => [220, 0, nil, nil, nil, nil, nil],
      798 => [220, 0, nil, nil, nil, nil, nil],
      799 => [220, 0, nil, nil, nil, nil, nil],
      800 => [220, 0, nil, nil, nil, nil, nil],
      801 => [202, 0, nil, nil, nil, nil, nil],
      802 => [202, 0, nil, nil, nil, nil, nil],
      803 => [220, 0, nil, nil, nil, nil, nil],
      804 => [220, 0, nil, nil, nil, nil, nil],
      805 => [220, 0, nil, nil, nil, nil, nil],
      806 => [220, 0, nil, nil, nil, nil, nil],
      807 => [202, 0, nil, nil, nil, nil, nil],
      808 => [202, 0, nil, nil, nil, nil, nil],
      809 => [220, 0, nil, nil, nil, nil, nil],
      810 => [220, 0, nil, nil, nil, nil, nil],
      811 => [220, 0, nil, nil, nil, nil, nil],
      812 => [220, 0, nil, nil, nil, nil, nil],
      813 => [220, 0, nil, nil, nil, nil, nil],
      814 => [220, 0, nil, nil, nil, nil, nil],
      815 => [220, 0, nil, nil, nil, nil, nil],
      816 => [220, 0, nil, nil, nil, nil, nil],
      817 => [220, 0, nil, nil, nil, nil, nil],
      818 => [220, 0, nil, nil, nil, nil, nil],
      819 => [220, 0, nil, nil, nil, nil, nil],
      820 => [1, 0, nil, nil, nil, nil, nil],
      821 => [1, 0, nil, nil, nil, nil, nil],
      822 => [1, 0, nil, nil, nil, nil, nil],
      823 => [1, 0, nil, nil, nil, nil, nil],
      824 => [1, 0, nil, nil, nil, nil, nil],
      825 => [220, 0, nil, nil, nil, nil, nil],
      826 => [220, 0, nil, nil, nil, nil, nil],
      827 => [220, 0, nil, nil, nil, nil, nil],
      828 => [220, 0, nil, nil, nil, nil, nil],
      829 => [230, 0, nil, nil, nil, nil, nil],
      830 => [230, 0, nil, nil, nil, nil, nil],
      831 => [230, 0, nil, nil, nil, nil, nil],
      832 => [230, 2, "̀", "̀", nil, nil, nil],
      833 => [230, 2, "́", "́", nil, nil, nil],
      834 => [230, 0, nil, nil, nil, nil, nil],
      835 => [230, 2, "̓", "̓", nil, nil, nil],
      836 => [230, 3, "̈́", "̈́", nil, nil, nil],
      837 => [240, 0, nil, nil, 921, nil, 921],
      838 => [230, 0, nil, nil, nil, nil, nil],
      839 => [220, 0, nil, nil, nil, nil, nil],
      840 => [220, 0, nil, nil, nil, nil, nil],
      841 => [220, 0, nil, nil, nil, nil, nil],
      842 => [230, 0, nil, nil, nil, nil, nil],
      843 => [230, 0, nil, nil, nil, nil, nil],
      844 => [230, 0, nil, nil, nil, nil, nil],
      845 => [220, 0, nil, nil, nil, nil, nil],
      846 => [220, 0, nil, nil, nil, nil, nil],
      864 => [234, 0, nil, nil, nil, nil, nil],
      865 => [234, 0, nil, nil, nil, nil, nil],
      866 => [233, 0, nil, nil, nil, nil, nil],
      884 => [0, 2, "ʹ", "ʹ", nil, nil, nil],
      890 => [0, 0, nil, " ͅ", nil, nil, nil],
      894 => [0, 2, ";", ";", nil, nil, nil],
      900 => [0, 0, nil, " ́", nil, nil, nil],
      901 => [0, 0, "΅", "΅", nil, nil, nil],
      902 => [0, 0, "Ά", "Ά", nil, 940, nil],
      903 => [0, 2, "·", "·", nil, nil, nil],
      904 => [0, 0, "Έ", "Έ", nil, 941, nil],
      905 => [0, 0, "Ή", "Ή", nil, 942, nil],
      906 => [0, 0, "Ί", "Ί", nil, 943, nil],
      908 => [0, 0, "Ό", "Ό", nil, 972, nil],
      910 => [0, 0, "Ύ", "Ύ", nil, 973, nil],
      911 => [0, 0, "Ώ", "Ώ", nil, 974, nil],
      912 => [0, 0, "ΐ", "ΐ", nil, nil, nil],
      913 => [0, 0, nil, nil, nil, 945, nil],
      914 => [0, 0, nil, nil, nil, 946, nil],
      915 => [0, 0, nil, nil, nil, 947, nil],
      916 => [0, 0, nil, nil, nil, 948, nil],
      917 => [0, 0, nil, nil, nil, 949, nil],
      918 => [0, 0, nil, nil, nil, 950, nil],
      919 => [0, 0, nil, nil, nil, 951, nil],
      920 => [0, 0, nil, nil, nil, 952, nil],
      921 => [0, 0, nil, nil, nil, 953, nil],
      922 => [0, 0, nil, nil, nil, 954, nil],
      923 => [0, 0, nil, nil, nil, 955, nil],
      924 => [0, 0, nil, nil, nil, 956, nil],
      925 => [0, 0, nil, nil, nil, 957, nil],
      926 => [0, 0, nil, nil, nil, 958, nil],
      927 => [0, 0, nil, nil, nil, 959, nil],
      928 => [0, 0, nil, nil, nil, 960, nil],
      929 => [0, 0, nil, nil, nil, 961, nil],
      931 => [0, 0, nil, nil, nil, 963, nil],
      932 => [0, 0, nil, nil, nil, 964, nil],
      933 => [0, 0, nil, nil, nil, 965, nil],
      934 => [0, 0, nil, nil, nil, 966, nil],
      935 => [0, 0, nil, nil, nil, 967, nil],
      936 => [0, 0, nil, nil, nil, 968, nil],
      937 => [0, 0, nil, nil, nil, 969, nil],
      938 => [0, 0, "Ϊ", "Ϊ", nil, 970, nil],
      939 => [0, 0, "Ϋ", "Ϋ", nil, 971, nil],
      940 => [0, 0, "ά", "ά", 902, nil, 902],
      941 => [0, 0, "έ", "έ", 904, nil, 904],
      942 => [0, 0, "ή", "ή", 905, nil, 905],
      943 => [0, 0, "ί", "ί", 906, nil, 906],
      944 => [0, 0, "ΰ", "ΰ", nil, nil, nil],
      945 => [0, 0, nil, nil, 913, nil, 913],
      946 => [0, 0, nil, nil, 914, nil, 914],
      947 => [0, 0, nil, nil, 915, nil, 915],
      948 => [0, 0, nil, nil, 916, nil, 916],
      949 => [0, 0, nil, nil, 917, nil, 917],
      950 => [0, 0, nil, nil, 918, nil, 918],
      951 => [0, 0, nil, nil, 919, nil, 919],
      952 => [0, 0, nil, nil, 920, nil, 920],
      953 => [0, 0, nil, nil, 921, nil, 921],
      954 => [0, 0, nil, nil, 922, nil, 922],
      955 => [0, 0, nil, nil, 923, nil, 923],
      956 => [0, 0, nil, nil, 924, nil, 924],
      957 => [0, 0, nil, nil, 925, nil, 925],
      958 => [0, 0, nil, nil, 926, nil, 926],
      959 => [0, 0, nil, nil, 927, nil, 927],
      960 => [0, 0, nil, nil, 928, nil, 928],
      961 => [0, 0, nil, nil, 929, nil, 929],
      962 => [0, 0, nil, nil, 931, nil, 931],
      963 => [0, 0, nil, nil, 931, nil, 931],
      964 => [0, 0, nil, nil, 932, nil, 932],
      965 => [0, 0, nil, nil, 933, nil, 933],
      966 => [0, 0, nil, nil, 934, nil, 934],
      967 => [0, 0, nil, nil, 935, nil, 935],
      968 => [0, 0, nil, nil, 936, nil, 936],
      969 => [0, 0, nil, nil, 937, nil, 937],
      970 => [0, 0, "ϊ", "ϊ", 938, nil, 938],
      971 => [0, 0, "ϋ", "ϋ", 939, nil, 939],
      972 => [0, 0, "ό", "ό", 908, nil, 908],
      973 => [0, 0, "ύ", "ύ", 910, nil, 910],
      974 => [0, 0, "ώ", "ώ", 911, nil, 911],
      976 => [0, 0, nil, "β", 914, nil, 914],
      977 => [0, 0, nil, "θ", 920, nil, 920],
      978 => [0, 0, nil, "Υ", nil, nil, nil],
      979 => [0, 0, "ϓ", "ϓ", nil, nil, nil],
      980 => [0, 0, "ϔ", "ϔ", nil, nil, nil],
      981 => [0, 0, nil, "φ", 934, nil, 934],
      982 => [0, 0, nil, "π", 928, nil, 928],
      986 => [0, 0, nil, nil, nil, 987, nil],
      987 => [0, 0, nil, nil, 986, nil, 986],
      988 => [0, 0, nil, nil, nil, 989, nil],
      989 => [0, 0, nil, nil, 988, nil, 988],
      990 => [0, 0, nil, nil, nil, 991, nil],
      991 => [0, 0, nil, nil, 990, nil, 990],
      992 => [0, 0, nil, nil, nil, 993, nil],
      993 => [0, 0, nil, nil, 992, nil, 992],
      994 => [0, 0, nil, nil, nil, 995, nil],
      995 => [0, 0, nil, nil, 994, nil, 994],
      996 => [0, 0, nil, nil, nil, 997, nil],
      997 => [0, 0, nil, nil, 996, nil, 996],
      998 => [0, 0, nil, nil, nil, 999, nil],
      999 => [0, 0, nil, nil, 998, nil, 998],
      1000 => [0, 0, nil, nil, nil, 1001, nil],
      1001 => [0, 0, nil, nil, 1000, nil, 1000],
      1002 => [0, 0, nil, nil, nil, 1003, nil],
      1003 => [0, 0, nil, nil, 1002, nil, 1002],
      1004 => [0, 0, nil, nil, nil, 1005, nil],
      1005 => [0, 0, nil, nil, 1004, nil, 1004],
      1006 => [0, 0, nil, nil, nil, 1007, nil],
      1007 => [0, 0, nil, nil, 1006, nil, 1006],
      1008 => [0, 0, nil, "κ", 922, nil, 922],
      1009 => [0, 0, nil, "ρ", 929, nil, 929],
      1010 => [0, 0, nil, "ς", 931, nil, 931],
      1024 => [0, 0, "Ѐ", "Ѐ", nil, 1104, nil],
      1025 => [0, 0, "Ё", "Ё", nil, 1105, nil],
      1026 => [0, 0, nil, nil, nil, 1106, nil],
      1027 => [0, 0, "Ѓ", "Ѓ", nil, 1107, nil],
      1028 => [0, 0, nil, nil, nil, 1108, nil],
      1029 => [0, 0, nil, nil, nil, 1109, nil],
      1030 => [0, 0, nil, nil, nil, 1110, nil],
      1031 => [0, 0, "Ї", "Ї", nil, 1111, nil],
      1032 => [0, 0, nil, nil, nil, 1112, nil],
      1033 => [0, 0, nil, nil, nil, 1113, nil],
      1034 => [0, 0, nil, nil, nil, 1114, nil],
      1035 => [0, 0, nil, nil, nil, 1115, nil],
      1036 => [0, 0, "Ќ", "Ќ", nil, 1116, nil],
      1037 => [0, 0, "Ѝ", "Ѝ", nil, 1117, nil],
      1038 => [0, 0, "Ў", "Ў", nil, 1118, nil],
      1039 => [0, 0, nil, nil, nil, 1119, nil],
      1040 => [0, 0, nil, nil, nil, 1072, nil],
      1041 => [0, 0, nil, nil, nil, 1073, nil],
      1042 => [0, 0, nil, nil, nil, 1074, nil],
      1043 => [0, 0, nil, nil, nil, 1075, nil],
      1044 => [0, 0, nil, nil, nil, 1076, nil],
      1045 => [0, 0, nil, nil, nil, 1077, nil],
      1046 => [0, 0, nil, nil, nil, 1078, nil],
      1047 => [0, 0, nil, nil, nil, 1079, nil],
      1048 => [0, 0, nil, nil, nil, 1080, nil],
      1049 => [0, 0, "Й", "Й", nil, 1081, nil],
      1050 => [0, 0, nil, nil, nil, 1082, nil],
      1051 => [0, 0, nil, nil, nil, 1083, nil],
      1052 => [0, 0, nil, nil, nil, 1084, nil],
      1053 => [0, 0, nil, nil, nil, 1085, nil],
      1054 => [0, 0, nil, nil, nil, 1086, nil],
      1055 => [0, 0, nil, nil, nil, 1087, nil],
      1056 => [0, 0, nil, nil, nil, 1088, nil],
      1057 => [0, 0, nil, nil, nil, 1089, nil],
      1058 => [0, 0, nil, nil, nil, 1090, nil],
      1059 => [0, 0, nil, nil, nil, 1091, nil],
      1060 => [0, 0, nil, nil, nil, 1092, nil],
      1061 => [0, 0, nil, nil, nil, 1093, nil],
      1062 => [0, 0, nil, nil, nil, 1094, nil],
      1063 => [0, 0, nil, nil, nil, 1095, nil],
      1064 => [0, 0, nil, nil, nil, 1096, nil],
      1065 => [0, 0, nil, nil, nil, 1097, nil],
      1066 => [0, 0, nil, nil, nil, 1098, nil],
      1067 => [0, 0, nil, nil, nil, 1099, nil],
      1068 => [0, 0, nil, nil, nil, 1100, nil],
      1069 => [0, 0, nil, nil, nil, 1101, nil],
      1070 => [0, 0, nil, nil, nil, 1102, nil],
      1071 => [0, 0, nil, nil, nil, 1103, nil],
      1072 => [0, 0, nil, nil, 1040, nil, 1040],
      1073 => [0, 0, nil, nil, 1041, nil, 1041],
      1074 => [0, 0, nil, nil, 1042, nil, 1042],
      1075 => [0, 0, nil, nil, 1043, nil, 1043],
      1076 => [0, 0, nil, nil, 1044, nil, 1044],
      1077 => [0, 0, nil, nil, 1045, nil, 1045],
      1078 => [0, 0, nil, nil, 1046, nil, 1046],
      1079 => [0, 0, nil, nil, 1047, nil, 1047],
      1080 => [0, 0, nil, nil, 1048, nil, 1048],
      1081 => [0, 0, "й", "й", 1049, nil, 1049],
      1082 => [0, 0, nil, nil, 1050, nil, 1050],
      1083 => [0, 0, nil, nil, 1051, nil, 1051],
      1084 => [0, 0, nil, nil, 1052, nil, 1052],
      1085 => [0, 0, nil, nil, 1053, nil, 1053],
      1086 => [0, 0, nil, nil, 1054, nil, 1054],
      1087 => [0, 0, nil, nil, 1055, nil, 1055],
      1088 => [0, 0, nil, nil, 1056, nil, 1056],
      1089 => [0, 0, nil, nil, 1057, nil, 1057],
      1090 => [0, 0, nil, nil, 1058, nil, 1058],
      1091 => [0, 0, nil, nil, 1059, nil, 1059],
      1092 => [0, 0, nil, nil, 1060, nil, 1060],
      1093 => [0, 0, nil, nil, 1061, nil, 1061],
      1094 => [0, 0, nil, nil, 1062, nil, 1062],
      1095 => [0, 0, nil, nil, 1063, nil, 1063],
      1096 => [0, 0, nil, nil, 1064, nil, 1064],
      1097 => [0, 0, nil, nil, 1065, nil, 1065],
      1098 => [0, 0, nil, nil, 1066, nil, 1066],
      1099 => [0, 0, nil, nil, 1067, nil, 1067],
      1100 => [0, 0, nil, nil, 1068, nil, 1068],
      1101 => [0, 0, nil, nil, 1069, nil, 1069],
      1102 => [0, 0, nil, nil, 1070, nil, 1070],
      1103 => [0, 0, nil, nil, 1071, nil, 1071],
      1104 => [0, 0, "ѐ", "ѐ", 1024, nil, 1024],
      1105 => [0, 0, "ё", "ё", 1025, nil, 1025],
      1106 => [0, 0, nil, nil, 1026, nil, 1026],
      1107 => [0, 0, "ѓ", "ѓ", 1027, nil, 1027],
      1108 => [0, 0, nil, nil, 1028, nil, 1028],
      1109 => [0, 0, nil, nil, 1029, nil, 1029],
      1110 => [0, 0, nil, nil, 1030, nil, 1030],
      1111 => [0, 0, "ї", "ї", 1031, nil, 1031],
      1112 => [0, 0, nil, nil, 1032, nil, 1032],
      1113 => [0, 0, nil, nil, 1033, nil, 1033],
      1114 => [0, 0, nil, nil, 1034, nil, 1034],
      1115 => [0, 0, nil, nil, 1035, nil, 1035],
      1116 => [0, 0, "ќ", "ќ", 1036, nil, 1036],
      1117 => [0, 0, "ѝ", "ѝ", 1037, nil, 1037],
      1118 => [0, 0, "ў", "ў", 1038, nil, 1038],
      1119 => [0, 0, nil, nil, 1039, nil, 1039],
      1120 => [0, 0, nil, nil, nil, 1121, nil],
      1121 => [0, 0, nil, nil, 1120, nil, 1120],
      1122 => [0, 0, nil, nil, nil, 1123, nil],
      1123 => [0, 0, nil, nil, 1122, nil, 1122],
      1124 => [0, 0, nil, nil, nil, 1125, nil],
      1125 => [0, 0, nil, nil, 1124, nil, 1124],
      1126 => [0, 0, nil, nil, nil, 1127, nil],
      1127 => [0, 0, nil, nil, 1126, nil, 1126],
      1128 => [0, 0, nil, nil, nil, 1129, nil],
      1129 => [0, 0, nil, nil, 1128, nil, 1128],
      1130 => [0, 0, nil, nil, nil, 1131, nil],
      1131 => [0, 0, nil, nil, 1130, nil, 1130],
      1132 => [0, 0, nil, nil, nil, 1133, nil],
      1133 => [0, 0, nil, nil, 1132, nil, 1132],
      1134 => [0, 0, nil, nil, nil, 1135, nil],
      1135 => [0, 0, nil, nil, 1134, nil, 1134],
      1136 => [0, 0, nil, nil, nil, 1137, nil],
      1137 => [0, 0, nil, nil, 1136, nil, 1136],
      1138 => [0, 0, nil, nil, nil, 1139, nil],
      1139 => [0, 0, nil, nil, 1138, nil, 1138],
      1140 => [0, 0, nil, nil, nil, 1141, nil],
      1141 => [0, 0, nil, nil, 1140, nil, 1140],
      1142 => [0, 0, "Ѷ", "Ѷ", nil, 1143, nil],
      1143 => [0, 0, "ѷ", "ѷ", 1142, nil, 1142],
      1144 => [0, 0, nil, nil, nil, 1145, nil],
      1145 => [0, 0, nil, nil, 1144, nil, 1144],
      1146 => [0, 0, nil, nil, nil, 1147, nil],
      1147 => [0, 0, nil, nil, 1146, nil, 1146],
      1148 => [0, 0, nil, nil, nil, 1149, nil],
      1149 => [0, 0, nil, nil, 1148, nil, 1148],
      1150 => [0, 0, nil, nil, nil, 1151, nil],
      1151 => [0, 0, nil, nil, 1150, nil, 1150],
      1152 => [0, 0, nil, nil, nil, 1153, nil],
      1153 => [0, 0, nil, nil, 1152, nil, 1152],
      1155 => [230, 0, nil, nil, nil, nil, nil],
      1156 => [230, 0, nil, nil, nil, nil, nil],
      1157 => [230, 0, nil, nil, nil, nil, nil],
      1158 => [230, 0, nil, nil, nil, nil, nil],
      1164 => [0, 0, nil, nil, nil, 1165, nil],
      1165 => [0, 0, nil, nil, 1164, nil, 1164],
      1166 => [0, 0, nil, nil, nil, 1167, nil],
      1167 => [0, 0, nil, nil, 1166, nil, 1166],
      1168 => [0, 0, nil, nil, nil, 1169, nil],
      1169 => [0, 0, nil, nil, 1168, nil, 1168],
      1170 => [0, 0, nil, nil, nil, 1171, nil],
      1171 => [0, 0, nil, nil, 1170, nil, 1170],
      1172 => [0, 0, nil, nil, nil, 1173, nil],
      1173 => [0, 0, nil, nil, 1172, nil, 1172],
      1174 => [0, 0, nil, nil, nil, 1175, nil],
      1175 => [0, 0, nil, nil, 1174, nil, 1174],
      1176 => [0, 0, nil, nil, nil, 1177, nil],
      1177 => [0, 0, nil, nil, 1176, nil, 1176],
      1178 => [0, 0, nil, nil, nil, 1179, nil],
      1179 => [0, 0, nil, nil, 1178, nil, 1178],
      1180 => [0, 0, nil, nil, nil, 1181, nil],
      1181 => [0, 0, nil, nil, 1180, nil, 1180],
      1182 => [0, 0, nil, nil, nil, 1183, nil],
      1183 => [0, 0, nil, nil, 1182, nil, 1182],
      1184 => [0, 0, nil, nil, nil, 1185, nil],
      1185 => [0, 0, nil, nil, 1184, nil, 1184],
      1186 => [0, 0, nil, nil, nil, 1187, nil],
      1187 => [0, 0, nil, nil, 1186, nil, 1186],
      1188 => [0, 0, nil, nil, nil, 1189, nil],
      1189 => [0, 0, nil, nil, 1188, nil, 1188],
      1190 => [0, 0, nil, nil, nil, 1191, nil],
      1191 => [0, 0, nil, nil, 1190, nil, 1190],
      1192 => [0, 0, nil, nil, nil, 1193, nil],
      1193 => [0, 0, nil, nil, 1192, nil, 1192],
      1194 => [0, 0, nil, nil, nil, 1195, nil],
      1195 => [0, 0, nil, nil, 1194, nil, 1194],
      1196 => [0, 0, nil, nil, nil, 1197, nil],
      1197 => [0, 0, nil, nil, 1196, nil, 1196],
      1198 => [0, 0, nil, nil, nil, 1199, nil],
      1199 => [0, 0, nil, nil, 1198, nil, 1198],
      1200 => [0, 0, nil, nil, nil, 1201, nil],
      1201 => [0, 0, nil, nil, 1200, nil, 1200],
      1202 => [0, 0, nil, nil, nil, 1203, nil],
      1203 => [0, 0, nil, nil, 1202, nil, 1202],
      1204 => [0, 0, nil, nil, nil, 1205, nil],
      1205 => [0, 0, nil, nil, 1204, nil, 1204],
      1206 => [0, 0, nil, nil, nil, 1207, nil],
      1207 => [0, 0, nil, nil, 1206, nil, 1206],
      1208 => [0, 0, nil, nil, nil, 1209, nil],
      1209 => [0, 0, nil, nil, 1208, nil, 1208],
      1210 => [0, 0, nil, nil, nil, 1211, nil],
      1211 => [0, 0, nil, nil, 1210, nil, 1210],
      1212 => [0, 0, nil, nil, nil, 1213, nil],
      1213 => [0, 0, nil, nil, 1212, nil, 1212],
      1214 => [0, 0, nil, nil, nil, 1215, nil],
      1215 => [0, 0, nil, nil, 1214, nil, 1214],
      1217 => [0, 0, "Ӂ", "Ӂ", nil, 1218, nil],
      1218 => [0, 0, "ӂ", "ӂ", 1217, nil, 1217],
      1219 => [0, 0, nil, nil, nil, 1220, nil],
      1220 => [0, 0, nil, nil, 1219, nil, 1219],
      1223 => [0, 0, nil, nil, nil, 1224, nil],
      1224 => [0, 0, nil, nil, 1223, nil, 1223],
      1227 => [0, 0, nil, nil, nil, 1228, nil],
      1228 => [0, 0, nil, nil, 1227, nil, 1227],
      1232 => [0, 0, "Ӑ", "Ӑ", nil, 1233, nil],
      1233 => [0, 0, "ӑ", "ӑ", 1232, nil, 1232],
      1234 => [0, 0, "Ӓ", "Ӓ", nil, 1235, nil],
      1235 => [0, 0, "ӓ", "ӓ", 1234, nil, 1234],
      1236 => [0, 0, nil, nil, nil, 1237, nil],
      1237 => [0, 0, nil, nil, 1236, nil, 1236],
      1238 => [0, 0, "Ӗ", "Ӗ", nil, 1239, nil],
      1239 => [0, 0, "ӗ", "ӗ", 1238, nil, 1238],
      1240 => [0, 0, nil, nil, nil, 1241, nil],
      1241 => [0, 0, nil, nil, 1240, nil, 1240],
      1242 => [0, 0, "Ӛ", "Ӛ", nil, 1243, nil],
      1243 => [0, 0, "ӛ", "ӛ", 1242, nil, 1242],
      1244 => [0, 0, "Ӝ", "Ӝ", nil, 1245, nil],
      1245 => [0, 0, "ӝ", "ӝ", 1244, nil, 1244],
      1246 => [0, 0, "Ӟ", "Ӟ", nil, 1247, nil],
      1247 => [0, 0, "ӟ", "ӟ", 1246, nil, 1246],
      1248 => [0, 0, nil, nil, nil, 1249, nil],
      1249 => [0, 0, nil, nil, 1248, nil, 1248],
      1250 => [0, 0, "Ӣ", "Ӣ", nil, 1251, nil],
      1251 => [0, 0, "ӣ", "ӣ", 1250, nil, 1250],
      1252 => [0, 0, "Ӥ", "Ӥ", nil, 1253, nil],
      1253 => [0, 0, "ӥ", "ӥ", 1252, nil, 1252],
      1254 => [0, 0, "Ӧ", "Ӧ", nil, 1255, nil],
      1255 => [0, 0, "ӧ", "ӧ", 1254, nil, 1254],
      1256 => [0, 0, nil, nil, nil, 1257, nil],
      1257 => [0, 0, nil, nil, 1256, nil, 1256],
      1258 => [0, 0, "Ӫ", "Ӫ", nil, 1259, nil],
      1259 => [0, 0, "ӫ", "ӫ", 1258, nil, 1258],
      1260 => [0, 0, "Ӭ", "Ӭ", nil, 1261, nil],
      1261 => [0, 0, "ӭ", "ӭ", 1260, nil, 1260],
      1262 => [0, 0, "Ӯ", "Ӯ", nil, 1263, nil],
      1263 => [0, 0, "ӯ", "ӯ", 1262, nil, 1262],
      1264 => [0, 0, "Ӱ", "Ӱ", nil, 1265, nil],
      1265 => [0, 0, "ӱ", "ӱ", 1264, nil, 1264],
      1266 => [0, 0, "Ӳ", "Ӳ", nil, 1267, nil],
      1267 => [0, 0, "ӳ", "ӳ", 1266, nil, 1266],
      1268 => [0, 0, "Ӵ", "Ӵ", nil, 1269, nil],
      1269 => [0, 0, "ӵ", "ӵ", 1268, nil, 1268],
      1272 => [0, 0, "Ӹ", "Ӹ", nil, 1273, nil],
      1273 => [0, 0, "ӹ", "ӹ", 1272, nil, 1272],
      1329 => [0, 0, nil, nil, nil, 1377, nil],
      1330 => [0, 0, nil, nil, nil, 1378, nil],
      1331 => [0, 0, nil, nil, nil, 1379, nil],
      1332 => [0, 0, nil, nil, nil, 1380, nil],
      1333 => [0, 0, nil, nil, nil, 1381, nil],
      1334 => [0, 0, nil, nil, nil, 1382, nil],
      1335 => [0, 0, nil, nil, nil, 1383, nil],
      1336 => [0, 0, nil, nil, nil, 1384, nil],
      1337 => [0, 0, nil, nil, nil, 1385, nil],
      1338 => [0, 0, nil, nil, nil, 1386, nil],
      1339 => [0, 0, nil, nil, nil, 1387, nil],
      1340 => [0, 0, nil, nil, nil, 1388, nil],
      1341 => [0, 0, nil, nil, nil, 1389, nil],
      1342 => [0, 0, nil, nil, nil, 1390, nil],
      1343 => [0, 0, nil, nil, nil, 1391, nil],
      1344 => [0, 0, nil, nil, nil, 1392, nil],
      1345 => [0, 0, nil, nil, nil, 1393, nil],
      1346 => [0, 0, nil, nil, nil, 1394, nil],
      1347 => [0, 0, nil, nil, nil, 1395, nil],
      1348 => [0, 0, nil, nil, nil, 1396, nil],
      1349 => [0, 0, nil, nil, nil, 1397, nil],
      1350 => [0, 0, nil, nil, nil, 1398, nil],
      1351 => [0, 0, nil, nil, nil, 1399, nil],
      1352 => [0, 0, nil, nil, nil, 1400, nil],
      1353 => [0, 0, nil, nil, nil, 1401, nil],
      1354 => [0, 0, nil, nil, nil, 1402, nil],
      1355 => [0, 0, nil, nil, nil, 1403, nil],
      1356 => [0, 0, nil, nil, nil, 1404, nil],
      1357 => [0, 0, nil, nil, nil, 1405, nil],
      1358 => [0, 0, nil, nil, nil, 1406, nil],
      1359 => [0, 0, nil, nil, nil, 1407, nil],
      1360 => [0, 0, nil, nil, nil, 1408, nil],
      1361 => [0, 0, nil, nil, nil, 1409, nil],
      1362 => [0, 0, nil, nil, nil, 1410, nil],
      1363 => [0, 0, nil, nil, nil, 1411, nil],
      1364 => [0, 0, nil, nil, nil, 1412, nil],
      1365 => [0, 0, nil, nil, nil, 1413, nil],
      1366 => [0, 0, nil, nil, nil, 1414, nil],
      1377 => [0, 0, nil, nil, 1329, nil, 1329],
      1378 => [0, 0, nil, nil, 1330, nil, 1330],
      1379 => [0, 0, nil, nil, 1331, nil, 1331],
      1380 => [0, 0, nil, nil, 1332, nil, 1332],
      1381 => [0, 0, nil, nil, 1333, nil, 1333],
      1382 => [0, 0, nil, nil, 1334, nil, 1334],
      1383 => [0, 0, nil, nil, 1335, nil, 1335],
      1384 => [0, 0, nil, nil, 1336, nil, 1336],
      1385 => [0, 0, nil, nil, 1337, nil, 1337],
      1386 => [0, 0, nil, nil, 1338, nil, 1338],
      1387 => [0, 0, nil, nil, 1339, nil, 1339],
      1388 => [0, 0, nil, nil, 1340, nil, 1340],
      1389 => [0, 0, nil, nil, 1341, nil, 1341],
      1390 => [0, 0, nil, nil, 1342, nil, 1342],
      1391 => [0, 0, nil, nil, 1343, nil, 1343],
      1392 => [0, 0, nil, nil, 1344, nil, 1344],
      1393 => [0, 0, nil, nil, 1345, nil, 1345],
      1394 => [0, 0, nil, nil, 1346, nil, 1346],
      1395 => [0, 0, nil, nil, 1347, nil, 1347],
      1396 => [0, 0, nil, nil, 1348, nil, 1348],
      1397 => [0, 0, nil, nil, 1349, nil, 1349],
      1398 => [0, 0, nil, nil, 1350, nil, 1350],
      1399 => [0, 0, nil, nil, 1351, nil, 1351],
      1400 => [0, 0, nil, nil, 1352, nil, 1352],
      1401 => [0, 0, nil, nil, 1353, nil, 1353],
      1402 => [0, 0, nil, nil, 1354, nil, 1354],
      1403 => [0, 0, nil, nil, 1355, nil, 1355],
      1404 => [0, 0, nil, nil, 1356, nil, 1356],
      1405 => [0, 0, nil, nil, 1357, nil, 1357],
      1406 => [0, 0, nil, nil, 1358, nil, 1358],
      1407 => [0, 0, nil, nil, 1359, nil, 1359],
      1408 => [0, 0, nil, nil, 1360, nil, 1360],
      1409 => [0, 0, nil, nil, 1361, nil, 1361],
      1410 => [0, 0, nil, nil, 1362, nil, 1362],
      1411 => [0, 0, nil, nil, 1363, nil, 1363],
      1412 => [0, 0, nil, nil, 1364, nil, 1364],
      1413 => [0, 0, nil, nil, 1365, nil, 1365],
      1414 => [0, 0, nil, nil, 1366, nil, 1366],
      1415 => [0, 0, nil, "եւ", nil, nil, nil],
      1425 => [220, 0, nil, nil, nil, nil, nil],
      1426 => [230, 0, nil, nil, nil, nil, nil],
      1427 => [230, 0, nil, nil, nil, nil, nil],
      1428 => [230, 0, nil, nil, nil, nil, nil],
      1429 => [230, 0, nil, nil, nil, nil, nil],
      1430 => [220, 0, nil, nil, nil, nil, nil],
      1431 => [230, 0, nil, nil, nil, nil, nil],
      1432 => [230, 0, nil, nil, nil, nil, nil],
      1433 => [230, 0, nil, nil, nil, nil, nil],
      1434 => [222, 0, nil, nil, nil, nil, nil],
      1435 => [220, 0, nil, nil, nil, nil, nil],
      1436 => [230, 0, nil, nil, nil, nil, nil],
      1437 => [230, 0, nil, nil, nil, nil, nil],
      1438 => [230, 0, nil, nil, nil, nil, nil],
      1439 => [230, 0, nil, nil, nil, nil, nil],
      1440 => [230, 0, nil, nil, nil, nil, nil],
      1441 => [230, 0, nil, nil, nil, nil, nil],
      1443 => [220, 0, nil, nil, nil, nil, nil],
      1444 => [220, 0, nil, nil, nil, nil, nil],
      1445 => [220, 0, nil, nil, nil, nil, nil],
      1446 => [220, 0, nil, nil, nil, nil, nil],
      1447 => [220, 0, nil, nil, nil, nil, nil],
      1448 => [230, 0, nil, nil, nil, nil, nil],
      1449 => [230, 0, nil, nil, nil, nil, nil],
      1450 => [220, 0, nil, nil, nil, nil, nil],
      1451 => [230, 0, nil, nil, nil, nil, nil],
      1452 => [230, 0, nil, nil, nil, nil, nil],
      1453 => [222, 0, nil, nil, nil, nil, nil],
      1454 => [228, 0, nil, nil, nil, nil, nil],
      1455 => [230, 0, nil, nil, nil, nil, nil],
      1456 => [10, 0, nil, nil, nil, nil, nil],
      1457 => [11, 0, nil, nil, nil, nil, nil],
      1458 => [12, 0, nil, nil, nil, nil, nil],
      1459 => [13, 0, nil, nil, nil, nil, nil],
      1460 => [14, 0, nil, nil, nil, nil, nil],
      1461 => [15, 0, nil, nil, nil, nil, nil],
      1462 => [16, 0, nil, nil, nil, nil, nil],
      1463 => [17, 0, nil, nil, nil, nil, nil],
      1464 => [18, 0, nil, nil, nil, nil, nil],
      1465 => [19, 0, nil, nil, nil, nil, nil],
      1467 => [20, 0, nil, nil, nil, nil, nil],
      1468 => [21, 0, nil, nil, nil, nil, nil],
      1469 => [22, 0, nil, nil, nil, nil, nil],
      1471 => [23, 0, nil, nil, nil, nil, nil],
      1473 => [24, 0, nil, nil, nil, nil, nil],
      1474 => [25, 0, nil, nil, nil, nil, nil],
      1476 => [230, 0, nil, nil, nil, nil, nil],
      1570 => [0, 0, "آ", "آ", nil, nil, nil],
      1571 => [0, 0, "أ", "أ", nil, nil, nil],
      1572 => [0, 0, "ؤ", "ؤ", nil, nil, nil],
      1573 => [0, 0, "إ", "إ", nil, nil, nil],
      1574 => [0, 0, "ئ", "ئ", nil, nil, nil],
      1611 => [27, 0, nil, nil, nil, nil, nil],
      1612 => [28, 0, nil, nil, nil, nil, nil],
      1613 => [29, 0, nil, nil, nil, nil, nil],
      1614 => [30, 0, nil, nil, nil, nil, nil],
      1615 => [31, 0, nil, nil, nil, nil, nil],
      1616 => [32, 0, nil, nil, nil, nil, nil],
      1617 => [33, 0, nil, nil, nil, nil, nil],
      1618 => [34, 0, nil, nil, nil, nil, nil],
      1619 => [230, 0, nil, nil, nil, nil, nil],
      1620 => [230, 0, nil, nil, nil, nil, nil],
      1621 => [220, 0, nil, nil, nil, nil, nil],
      1648 => [35, 0, nil, nil, nil, nil, nil],
      1653 => [0, 0, nil, "اٴ", nil, nil, nil],
      1654 => [0, 0, nil, "وٴ", nil, nil, nil],
      1655 => [0, 0, nil, "ۇٴ", nil, nil, nil],
      1656 => [0, 0, nil, "يٴ", nil, nil, nil],
      1728 => [0, 0, "ۀ", "ۀ", nil, nil, nil],
      1730 => [0, 0, "ۂ", "ۂ", nil, nil, nil],
      1747 => [0, 0, "ۓ", "ۓ", nil, nil, nil],
      1750 => [230, 0, nil, nil, nil, nil, nil],
      1751 => [230, 0, nil, nil, nil, nil, nil],
      1752 => [230, 0, nil, nil, nil, nil, nil],
      1753 => [230, 0, nil, nil, nil, nil, nil],
      1754 => [230, 0, nil, nil, nil, nil, nil],
      1755 => [230, 0, nil, nil, nil, nil, nil],
      1756 => [230, 0, nil, nil, nil, nil, nil],
      1759 => [230, 0, nil, nil, nil, nil, nil],
      1760 => [230, 0, nil, nil, nil, nil, nil],
      1761 => [230, 0, nil, nil, nil, nil, nil],
      1762 => [230, 0, nil, nil, nil, nil, nil],
      1763 => [220, 0, nil, nil, nil, nil, nil],
      1764 => [230, 0, nil, nil, nil, nil, nil],
      1767 => [230, 0, nil, nil, nil, nil, nil],
      1768 => [230, 0, nil, nil, nil, nil, nil],
      1770 => [220, 0, nil, nil, nil, nil, nil],
      1771 => [230, 0, nil, nil, nil, nil, nil],
      1772 => [230, 0, nil, nil, nil, nil, nil],
      1773 => [220, 0, nil, nil, nil, nil, nil],
      1809 => [36, 0, nil, nil, nil, nil, nil],
      1840 => [230, 0, nil, nil, nil, nil, nil],
      1841 => [220, 0, nil, nil, nil, nil, nil],
      1842 => [230, 0, nil, nil, nil, nil, nil],
      1843 => [230, 0, nil, nil, nil, nil, nil],
      1844 => [220, 0, nil, nil, nil, nil, nil],
      1845 => [230, 0, nil, nil, nil, nil, nil],
      1846 => [230, 0, nil, nil, nil, nil, nil],
      1847 => [220, 0, nil, nil, nil, nil, nil],
      1848 => [220, 0, nil, nil, nil, nil, nil],
      1849 => [220, 0, nil, nil, nil, nil, nil],
      1850 => [230, 0, nil, nil, nil, nil, nil],
      1851 => [220, 0, nil, nil, nil, nil, nil],
      1852 => [220, 0, nil, nil, nil, nil, nil],
      1853 => [230, 0, nil, nil, nil, nil, nil],
      1854 => [220, 0, nil, nil, nil, nil, nil],
      1855 => [230, 0, nil, nil, nil, nil, nil],
      1856 => [230, 0, nil, nil, nil, nil, nil],
      1857 => [230, 0, nil, nil, nil, nil, nil],
      1858 => [220, 0, nil, nil, nil, nil, nil],
      1859 => [230, 0, nil, nil, nil, nil, nil],
      1860 => [220, 0, nil, nil, nil, nil, nil],
      1861 => [230, 0, nil, nil, nil, nil, nil],
      1862 => [220, 0, nil, nil, nil, nil, nil],
      1863 => [230, 0, nil, nil, nil, nil, nil],
      1864 => [220, 0, nil, nil, nil, nil, nil],
      1865 => [230, 0, nil, nil, nil, nil, nil],
      1866 => [230, 0, nil, nil, nil, nil, nil],
      2345 => [0, 0, "ऩ", "ऩ", nil, nil, nil],
      2353 => [0, 0, "ऱ", "ऱ", nil, nil, nil],
      2356 => [0, 0, "ऴ", "ऴ", nil, nil, nil],
      2364 => [7, 0, nil, nil, nil, nil, nil],
      2381 => [9, 0, nil, nil, nil, nil, nil],
      2385 => [230, 0, nil, nil, nil, nil, nil],
      2386 => [220, 0, nil, nil, nil, nil, nil],
      2387 => [230, 0, nil, nil, nil, nil, nil],
      2388 => [230, 0, nil, nil, nil, nil, nil],
      2392 => [0, 1, "क़", "क़", nil, nil, nil],
      2393 => [0, 1, "ख़", "ख़", nil, nil, nil],
      2394 => [0, 1, "ग़", "ग़", nil, nil, nil],
      2395 => [0, 1, "ज़", "ज़", nil, nil, nil],
      2396 => [0, 1, "ड़", "ड़", nil, nil, nil],
      2397 => [0, 1, "ढ़", "ढ़", nil, nil, nil],
      2398 => [0, 1, "फ़", "फ़", nil, nil, nil],
      2399 => [0, 1, "य़", "य़", nil, nil, nil],
      2492 => [7, 0, nil, nil, nil, nil, nil],
      2507 => [0, 0, "ো", "ো", nil, nil, nil],
      2508 => [0, 0, "ৌ", "ৌ", nil, nil, nil],
      2509 => [9, 0, nil, nil, nil, nil, nil],
      2524 => [0, 1, "ড়", "ড়", nil, nil, nil],
      2525 => [0, 1, "ঢ়", "ঢ়", nil, nil, nil],
      2527 => [0, 1, "য়", "য়", nil, nil, nil],
      2611 => [0, 1, "ਲ਼", "ਲ਼", nil, nil, nil],
      2614 => [0, 1, "ਸ਼", "ਸ਼", nil, nil, nil],
      2620 => [7, 0, nil, nil, nil, nil, nil],
      2637 => [9, 0, nil, nil, nil, nil, nil],
      2649 => [0, 1, "ਖ਼", "ਖ਼", nil, nil, nil],
      2650 => [0, 1, "ਗ਼", "ਗ਼", nil, nil, nil],
      2651 => [0, 1, "ਜ਼", "ਜ਼", nil, nil, nil],
      2654 => [0, 1, "ਫ਼", "ਫ਼", nil, nil, nil],
      2748 => [7, 0, nil, nil, nil, nil, nil],
      2765 => [9, 0, nil, nil, nil, nil, nil],
      2876 => [7, 0, nil, nil, nil, nil, nil],
      2888 => [0, 0, "ୈ", "ୈ", nil, nil, nil],
      2891 => [0, 0, "ୋ", "ୋ", nil, nil, nil],
      2892 => [0, 0, "ୌ", "ୌ", nil, nil, nil],
      2893 => [9, 0, nil, nil, nil, nil, nil],
      2908 => [0, 1, "ଡ଼", "ଡ଼", nil, nil, nil],
      2909 => [0, 1, "ଢ଼", "ଢ଼", nil, nil, nil],
      2964 => [0, 0, "ஔ", "ஔ", nil, nil, nil],
      3018 => [0, 0, "ொ", "ொ", nil, nil, nil],
      3019 => [0, 0, "ோ", "ோ", nil, nil, nil],
      3020 => [0, 0, "ௌ", "ௌ", nil, nil, nil],
      3021 => [9, 0, nil, nil, nil, nil, nil],
      3144 => [0, 0, "ై", "ై", nil, nil, nil],
      3149 => [9, 0, nil, nil, nil, nil, nil],
      3157 => [84, 0, nil, nil, nil, nil, nil],
      3158 => [91, 0, nil, nil, nil, nil, nil],
      3264 => [0, 0, "ೀ", "ೀ", nil, nil, nil],
      3271 => [0, 0, "ೇ", "ೇ", nil, nil, nil],
      3272 => [0, 0, "ೈ", "ೈ", nil, nil, nil],
      3274 => [0, 0, "ೊ", "ೊ", nil, nil, nil],
      3275 => [0, 0, "ೋ", "ೋ", nil, nil, nil],
      3277 => [9, 0, nil, nil, nil, nil, nil],
      3402 => [0, 0, "ൊ", "ൊ", nil, nil, nil],
      3403 => [0, 0, "ോ", "ോ", nil, nil, nil],
      3404 => [0, 0, "ൌ", "ൌ", nil, nil, nil],
      3405 => [9, 0, nil, nil, nil, nil, nil],
      3530 => [9, 0, nil, nil, nil, nil, nil],
      3546 => [0, 0, "ේ", "ේ", nil, nil, nil],
      3548 => [0, 0, "ො", "ො", nil, nil, nil],
      3549 => [0, 0, "ෝ", "ෝ", nil, nil, nil],
      3550 => [0, 0, "ෞ", "ෞ", nil, nil, nil],
      3635 => [0, 0, nil, "ํา", nil, nil, nil],
      3640 => [103, 0, nil, nil, nil, nil, nil],
      3641 => [103, 0, nil, nil, nil, nil, nil],
      3642 => [9, 0, nil, nil, nil, nil, nil],
      3656 => [107, 0, nil, nil, nil, nil, nil],
      3657 => [107, 0, nil, nil, nil, nil, nil],
      3658 => [107, 0, nil, nil, nil, nil, nil],
      3659 => [107, 0, nil, nil, nil, nil, nil],
      3763 => [0, 0, nil, "ໍາ", nil, nil, nil],
      3768 => [118, 0, nil, nil, nil, nil, nil],
      3769 => [118, 0, nil, nil, nil, nil, nil],
      3784 => [122, 0, nil, nil, nil, nil, nil],
      3785 => [122, 0, nil, nil, nil, nil, nil],
      3786 => [122, 0, nil, nil, nil, nil, nil],
      3787 => [122, 0, nil, nil, nil, nil, nil],
      3804 => [0, 0, nil, "ຫນ", nil, nil, nil],
      3805 => [0, 0, nil, "ຫມ", nil, nil, nil],
      3852 => [0, 0, nil, "་", nil, nil, nil],
      3864 => [220, 0, nil, nil, nil, nil, nil],
      3865 => [220, 0, nil, nil, nil, nil, nil],
      3893 => [220, 0, nil, nil, nil, nil, nil],
      3895 => [220, 0, nil, nil, nil, nil, nil],
      3897 => [216, 0, nil, nil, nil, nil, nil],
      3907 => [0, 1, "གྷ", "གྷ", nil, nil, nil],
      3917 => [0, 1, "ཌྷ", "ཌྷ", nil, nil, nil],
      3922 => [0, 1, "དྷ", "དྷ", nil, nil, nil],
      3927 => [0, 1, "བྷ", "བྷ", nil, nil, nil],
      3932 => [0, 1, "ཛྷ", "ཛྷ", nil, nil, nil],
      3945 => [0, 1, "ཀྵ", "ཀྵ", nil, nil, nil],
      3953 => [129, 0, nil, nil, nil, nil, nil],
      3954 => [130, 0, nil, nil, nil, nil, nil],
      3955 => [0, 3, "ཱི", "ཱི", nil, nil, nil],
      3956 => [132, 0, nil, nil, nil, nil, nil],
      3957 => [0, 3, "ཱུ", "ཱུ", nil, nil, nil],
      3958 => [0, 1, "ྲྀ", "ྲྀ", nil, nil, nil],
      3959 => [0, 0, nil, "ྲཱྀ", nil, nil, nil],
      3960 => [0, 1, "ླྀ", "ླྀ", nil, nil, nil],
      3961 => [0, 0, nil, "ླཱྀ", nil, nil, nil],
      3962 => [130, 0, nil, nil, nil, nil, nil],
      3963 => [130, 0, nil, nil, nil, nil, nil],
      3964 => [130, 0, nil, nil, nil, nil, nil],
      3965 => [130, 0, nil, nil, nil, nil, nil],
      3968 => [130, 0, nil, nil, nil, nil, nil],
      3969 => [0, 3, "ཱྀ", "ཱྀ", nil, nil, nil],
      3970 => [230, 0, nil, nil, nil, nil, nil],
      3971 => [230, 0, nil, nil, nil, nil, nil],
      3972 => [9, 0, nil, nil, nil, nil, nil],
      3974 => [230, 0, nil, nil, nil, nil, nil],
      3975 => [230, 0, nil, nil, nil, nil, nil],
      3987 => [0, 1, "ྒྷ", "ྒྷ", nil, nil, nil],
      3997 => [0, 1, "ྜྷ", "ྜྷ", nil, nil, nil],
      4002 => [0, 1, "ྡྷ", "ྡྷ", nil, nil, nil],
      4007 => [0, 1, "ྦྷ", "ྦྷ", nil, nil, nil],
      4012 => [0, 1, "ྫྷ", "ྫྷ", nil, nil, nil],
      4025 => [0, 1, "ྐྵ", "ྐྵ", nil, nil, nil],
      4038 => [220, 0, nil, nil, nil, nil, nil],
      4134 => [0, 0, "ဦ", "ဦ", nil, nil, nil],
      4151 => [7, 0, nil, nil, nil, nil, nil],
      4153 => [9, 0, nil, nil, nil, nil, nil],
      6098 => [9, 0, nil, nil, nil, nil, nil],
      6313 => [228, 0, nil, nil, nil, nil, nil],
      7680 => [0, 0, "Ḁ", "Ḁ", nil, 7681, nil],
      7681 => [0, 0, "ḁ", "ḁ", 7680, nil, 7680],
      7682 => [0, 0, "Ḃ", "Ḃ", nil, 7683, nil],
      7683 => [0, 0, "ḃ", "ḃ", 7682, nil, 7682],
      7684 => [0, 0, "Ḅ", "Ḅ", nil, 7685, nil],
      7685 => [0, 0, "ḅ", "ḅ", 7684, nil, 7684],
      7686 => [0, 0, "Ḇ", "Ḇ", nil, 7687, nil],
      7687 => [0, 0, "ḇ", "ḇ", 7686, nil, 7686],
      7688 => [0, 0, "Ḉ", "Ḉ", nil, 7689, nil],
      7689 => [0, 0, "ḉ", "ḉ", 7688, nil, 7688],
      7690 => [0, 0, "Ḋ", "Ḋ", nil, 7691, nil],
      7691 => [0, 0, "ḋ", "ḋ", 7690, nil, 7690],
      7692 => [0, 0, "Ḍ", "Ḍ", nil, 7693, nil],
      7693 => [0, 0, "ḍ", "ḍ", 7692, nil, 7692],
      7694 => [0, 0, "Ḏ", "Ḏ", nil, 7695, nil],
      7695 => [0, 0, "ḏ", "ḏ", 7694, nil, 7694],
      7696 => [0, 0, "Ḑ", "Ḑ", nil, 7697, nil],
      7697 => [0, 0, "ḑ", "ḑ", 7696, nil, 7696],
      7698 => [0, 0, "Ḓ", "Ḓ", nil, 7699, nil],
      7699 => [0, 0, "ḓ", "ḓ", 7698, nil, 7698],
      7700 => [0, 0, "Ḕ", "Ḕ", nil, 7701, nil],
      7701 => [0, 0, "ḕ", "ḕ", 7700, nil, 7700],
      7702 => [0, 0, "Ḗ", "Ḗ", nil, 7703, nil],
      7703 => [0, 0, "ḗ", "ḗ", 7702, nil, 7702],
      7704 => [0, 0, "Ḙ", "Ḙ", nil, 7705, nil],
      7705 => [0, 0, "ḙ", "ḙ", 7704, nil, 7704],
      7706 => [0, 0, "Ḛ", "Ḛ", nil, 7707, nil],
      7707 => [0, 0, "ḛ", "ḛ", 7706, nil, 7706],
      7708 => [0, 0, "Ḝ", "Ḝ", nil, 7709, nil],
      7709 => [0, 0, "ḝ", "ḝ", 7708, nil, 7708],
      7710 => [0, 0, "Ḟ", "Ḟ", nil, 7711, nil],
      7711 => [0, 0, "ḟ", "ḟ", 7710, nil, 7710],
      7712 => [0, 0, "Ḡ", "Ḡ", nil, 7713, nil],
      7713 => [0, 0, "ḡ", "ḡ", 7712, nil, 7712],
      7714 => [0, 0, "Ḣ", "Ḣ", nil, 7715, nil],
      7715 => [0, 0, "ḣ", "ḣ", 7714, nil, 7714],
      7716 => [0, 0, "Ḥ", "Ḥ", nil, 7717, nil],
      7717 => [0, 0, "ḥ", "ḥ", 7716, nil, 7716],
      7718 => [0, 0, "Ḧ", "Ḧ", nil, 7719, nil],
      7719 => [0, 0, "ḧ", "ḧ", 7718, nil, 7718],
      7720 => [0, 0, "Ḩ", "Ḩ", nil, 7721, nil],
      7721 => [0, 0, "ḩ", "ḩ", 7720, nil, 7720],
      7722 => [0, 0, "Ḫ", "Ḫ", nil, 7723, nil],
      7723 => [0, 0, "ḫ", "ḫ", 7722, nil, 7722],
      7724 => [0, 0, "Ḭ", "Ḭ", nil, 7725, nil],
      7725 => [0, 0, "ḭ", "ḭ", 7724, nil, 7724],
      7726 => [0, 0, "Ḯ", "Ḯ", nil, 7727, nil],
      7727 => [0, 0, "ḯ", "ḯ", 7726, nil, 7726],
      7728 => [0, 0, "Ḱ", "Ḱ", nil, 7729, nil],
      7729 => [0, 0, "ḱ", "ḱ", 7728, nil, 7728],
      7730 => [0, 0, "Ḳ", "Ḳ", nil, 7731, nil],
      7731 => [0, 0, "ḳ", "ḳ", 7730, nil, 7730],
      7732 => [0, 0, "Ḵ", "Ḵ", nil, 7733, nil],
      7733 => [0, 0, "ḵ", "ḵ", 7732, nil, 7732],
      7734 => [0, 0, "Ḷ", "Ḷ", nil, 7735, nil],
      7735 => [0, 0, "ḷ", "ḷ", 7734, nil, 7734],
      7736 => [0, 0, "Ḹ", "Ḹ", nil, 7737, nil],
      7737 => [0, 0, "ḹ", "ḹ", 7736, nil, 7736],
      7738 => [0, 0, "Ḻ", "Ḻ", nil, 7739, nil],
      7739 => [0, 0, "ḻ", "ḻ", 7738, nil, 7738],
      7740 => [0, 0, "Ḽ", "Ḽ", nil, 7741, nil],
      7741 => [0, 0, "ḽ", "ḽ", 7740, nil, 7740],
      7742 => [0, 0, "Ḿ", "Ḿ", nil, 7743, nil],
      7743 => [0, 0, "ḿ", "ḿ", 7742, nil, 7742],
      7744 => [0, 0, "Ṁ", "Ṁ", nil, 7745, nil],
      7745 => [0, 0, "ṁ", "ṁ", 7744, nil, 7744],
      7746 => [0, 0, "Ṃ", "Ṃ", nil, 7747, nil],
      7747 => [0, 0, "ṃ", "ṃ", 7746, nil, 7746],
      7748 => [0, 0, "Ṅ", "Ṅ", nil, 7749, nil],
      7749 => [0, 0, "ṅ", "ṅ", 7748, nil, 7748],
      7750 => [0, 0, "Ṇ", "Ṇ", nil, 7751, nil],
      7751 => [0, 0, "ṇ", "ṇ", 7750, nil, 7750],
      7752 => [0, 0, "Ṉ", "Ṉ", nil, 7753, nil],
      7753 => [0, 0, "ṉ", "ṉ", 7752, nil, 7752],
      7754 => [0, 0, "Ṋ", "Ṋ", nil, 7755, nil],
      7755 => [0, 0, "ṋ", "ṋ", 7754, nil, 7754],
      7756 => [0, 0, "Ṍ", "Ṍ", nil, 7757, nil],
      7757 => [0, 0, "ṍ", "ṍ", 7756, nil, 7756],
      7758 => [0, 0, "Ṏ", "Ṏ", nil, 7759, nil],
      7759 => [0, 0, "ṏ", "ṏ", 7758, nil, 7758],
      7760 => [0, 0, "Ṑ", "Ṑ", nil, 7761, nil],
      7761 => [0, 0, "ṑ", "ṑ", 7760, nil, 7760],
      7762 => [0, 0, "Ṓ", "Ṓ", nil, 7763, nil],
      7763 => [0, 0, "ṓ", "ṓ", 7762, nil, 7762],
      7764 => [0, 0, "Ṕ", "Ṕ", nil, 7765, nil],
      7765 => [0, 0, "ṕ", "ṕ", 7764, nil, 7764],
      7766 => [0, 0, "Ṗ", "Ṗ", nil, 7767, nil],
      7767 => [0, 0, "ṗ", "ṗ", 7766, nil, 7766],
      7768 => [0, 0, "Ṙ", "Ṙ", nil, 7769, nil],
      7769 => [0, 0, "ṙ", "ṙ", 7768, nil, 7768],
      7770 => [0, 0, "Ṛ", "Ṛ", nil, 7771, nil],
      7771 => [0, 0, "ṛ", "ṛ", 7770, nil, 7770],
      7772 => [0, 0, "Ṝ", "Ṝ", nil, 7773, nil],
      7773 => [0, 0, "ṝ", "ṝ", 7772, nil, 7772],
      7774 => [0, 0, "Ṟ", "Ṟ", nil, 7775, nil],
      7775 => [0, 0, "ṟ", "ṟ", 7774, nil, 7774],
      7776 => [0, 0, "Ṡ", "Ṡ", nil, 7777, nil],
      7777 => [0, 0, "ṡ", "ṡ", 7776, nil, 7776],
      7778 => [0, 0, "Ṣ", "Ṣ", nil, 7779, nil],
      7779 => [0, 0, "ṣ", "ṣ", 7778, nil, 7778],
      7780 => [0, 0, "Ṥ", "Ṥ", nil, 7781, nil],
      7781 => [0, 0, "ṥ", "ṥ", 7780, nil, 7780],
      7782 => [0, 0, "Ṧ", "Ṧ", nil, 7783, nil],
      7783 => [0, 0, "ṧ", "ṧ", 7782, nil, 7782],
      7784 => [0, 0, "Ṩ", "Ṩ", nil, 7785, nil],
      7785 => [0, 0, "ṩ", "ṩ", 7784, nil, 7784],
      7786 => [0, 0, "Ṫ", "Ṫ", nil, 7787, nil],
      7787 => [0, 0, "ṫ", "ṫ", 7786, nil, 7786],
      7788 => [0, 0, "Ṭ", "Ṭ", nil, 7789, nil],
      7789 => [0, 0, "ṭ", "ṭ", 7788, nil, 7788],
      7790 => [0, 0, "Ṯ", "Ṯ", nil, 7791, nil],
      7791 => [0, 0, "ṯ", "ṯ", 7790, nil, 7790],
      7792 => [0, 0, "Ṱ", "Ṱ", nil, 7793, nil],
      7793 => [0, 0, "ṱ", "ṱ", 7792, nil, 7792],
      7794 => [0, 0, "Ṳ", "Ṳ", nil, 7795, nil],
      7795 => [0, 0, "ṳ", "ṳ", 7794, nil, 7794],
      7796 => [0, 0, "Ṵ", "Ṵ", nil, 7797, nil],
      7797 => [0, 0, "ṵ", "ṵ", 7796, nil, 7796],
      7798 => [0, 0, "Ṷ", "Ṷ", nil, 7799, nil],
      7799 => [0, 0, "ṷ", "ṷ", 7798, nil, 7798],
      7800 => [0, 0, "Ṹ", "Ṹ", nil, 7801, nil],
      7801 => [0, 0, "ṹ", "ṹ", 7800, nil, 7800],
      7802 => [0, 0, "Ṻ", "Ṻ", nil, 7803, nil],
      7803 => [0, 0, "ṻ", "ṻ", 7802, nil, 7802],
      7804 => [0, 0, "Ṽ", "Ṽ", nil, 7805, nil],
      7805 => [0, 0, "ṽ", "ṽ", 7804, nil, 7804],
      7806 => [0, 0, "Ṿ", "Ṿ", nil, 7807, nil],
      7807 => [0, 0, "ṿ", "ṿ", 7806, nil, 7806],
      7808 => [0, 0, "Ẁ", "Ẁ", nil, 7809, nil],
      7809 => [0, 0, "ẁ", "ẁ", 7808, nil, 7808],
      7810 => [0, 0, "Ẃ", "Ẃ", nil, 7811, nil],
      7811 => [0, 0, "ẃ", "ẃ", 7810, nil, 7810],
      7812 => [0, 0, "Ẅ", "Ẅ", nil, 7813, nil],
      7813 => [0, 0, "ẅ", "ẅ", 7812, nil, 7812],
      7814 => [0, 0, "Ẇ", "Ẇ", nil, 7815, nil],
      7815 => [0, 0, "ẇ", "ẇ", 7814, nil, 7814],
      7816 => [0, 0, "Ẉ", "Ẉ", nil, 7817, nil],
      7817 => [0, 0, "ẉ", "ẉ", 7816, nil, 7816],
      7818 => [0, 0, "Ẋ", "Ẋ", nil, 7819, nil],
      7819 => [0, 0, "ẋ", "ẋ", 7818, nil, 7818],
      7820 => [0, 0, "Ẍ", "Ẍ", nil, 7821, nil],
      7821 => [0, 0, "ẍ", "ẍ", 7820, nil, 7820],
      7822 => [0, 0, "Ẏ", "Ẏ", nil, 7823, nil],
      7823 => [0, 0, "ẏ", "ẏ", 7822, nil, 7822],
      7824 => [0, 0, "Ẑ", "Ẑ", nil, 7825, nil],
      7825 => [0, 0, "ẑ", "ẑ", 7824, nil, 7824],
      7826 => [0, 0, "Ẓ", "Ẓ", nil, 7827, nil],
      7827 => [0, 0, "ẓ", "ẓ", 7826, nil, 7826],
      7828 => [0, 0, "Ẕ", "Ẕ", nil, 7829, nil],
      7829 => [0, 0, "ẕ", "ẕ", 7828, nil, 7828],
      7830 => [0, 0, "ẖ", "ẖ", nil, nil, nil],
      7831 => [0, 0, "ẗ", "ẗ", nil, nil, nil],
      7832 => [0, 0, "ẘ", "ẘ", nil, nil, nil],
      7833 => [0, 0, "ẙ", "ẙ", nil, nil, nil],
      7834 => [0, 0, nil, "aʾ", nil, nil, nil],
      7835 => [0, 0, "ẛ", "ẛ", 7776, nil, 7776],
      7840 => [0, 0, "Ạ", "Ạ", nil, 7841, nil],
      7841 => [0, 0, "ạ", "ạ", 7840, nil, 7840],
      7842 => [0, 0, "Ả", "Ả", nil, 7843, nil],
      7843 => [0, 0, "ả", "ả", 7842, nil, 7842],
      7844 => [0, 0, "Ấ", "Ấ", nil, 7845, nil],
      7845 => [0, 0, "ấ", "ấ", 7844, nil, 7844],
      7846 => [0, 0, "Ầ", "Ầ", nil, 7847, nil],
      7847 => [0, 0, "ầ", "ầ", 7846, nil, 7846],
      7848 => [0, 0, "Ẩ", "Ẩ", nil, 7849, nil],
      7849 => [0, 0, "ẩ", "ẩ", 7848, nil, 7848],
      7850 => [0, 0, "Ẫ", "Ẫ", nil, 7851, nil],
      7851 => [0, 0, "ẫ", "ẫ", 7850, nil, 7850],
      7852 => [0, 0, "Ậ", "Ậ", nil, 7853, nil],
      7853 => [0, 0, "ậ", "ậ", 7852, nil, 7852],
      7854 => [0, 0, "Ắ", "Ắ", nil, 7855, nil],
      7855 => [0, 0, "ắ", "ắ", 7854, nil, 7854],
      7856 => [0, 0, "Ằ", "Ằ", nil, 7857, nil],
      7857 => [0, 0, "ằ", "ằ", 7856, nil, 7856],
      7858 => [0, 0, "Ẳ", "Ẳ", nil, 7859, nil],
      7859 => [0, 0, "ẳ", "ẳ", 7858, nil, 7858],
      7860 => [0, 0, "Ẵ", "Ẵ", nil, 7861, nil],
      7861 => [0, 0, "ẵ", "ẵ", 7860, nil, 7860],
      7862 => [0, 0, "Ặ", "Ặ", nil, 7863, nil],
      7863 => [0, 0, "ặ", "ặ", 7862, nil, 7862],
      7864 => [0, 0, "Ẹ", "Ẹ", nil, 7865, nil],
      7865 => [0, 0, "ẹ", "ẹ", 7864, nil, 7864],
      7866 => [0, 0, "Ẻ", "Ẻ", nil, 7867, nil],
      7867 => [0, 0, "ẻ", "ẻ", 7866, nil, 7866],
      7868 => [0, 0, "Ẽ", "Ẽ", nil, 7869, nil],
      7869 => [0, 0, "ẽ", "ẽ", 7868, nil, 7868],
      7870 => [0, 0, "Ế", "Ế", nil, 7871, nil],
      7871 => [0, 0, "ế", "ế", 7870, nil, 7870],
      7872 => [0, 0, "Ề", "Ề", nil, 7873, nil],
      7873 => [0, 0, "ề", "ề", 7872, nil, 7872],
      7874 => [0, 0, "Ể", "Ể", nil, 7875, nil],
      7875 => [0, 0, "ể", "ể", 7874, nil, 7874],
      7876 => [0, 0, "Ễ", "Ễ", nil, 7877, nil],
      7877 => [0, 0, "ễ", "ễ", 7876, nil, 7876],
      7878 => [0, 0, "Ệ", "Ệ", nil, 7879, nil],
      7879 => [0, 0, "ệ", "ệ", 7878, nil, 7878],
      7880 => [0, 0, "Ỉ", "Ỉ", nil, 7881, nil],
      7881 => [0, 0, "ỉ", "ỉ", 7880, nil, 7880],
      7882 => [0, 0, "Ị", "Ị", nil, 7883, nil],
      7883 => [0, 0, "ị", "ị", 7882, nil, 7882],
      7884 => [0, 0, "Ọ", "Ọ", nil, 7885, nil],
      7885 => [0, 0, "ọ", "ọ", 7884, nil, 7884],
      7886 => [0, 0, "Ỏ", "Ỏ", nil, 7887, nil],
      7887 => [0, 0, "ỏ", "ỏ", 7886, nil, 7886],
      7888 => [0, 0, "Ố", "Ố", nil, 7889, nil],
      7889 => [0, 0, "ố", "ố", 7888, nil, 7888],
      7890 => [0, 0, "Ồ", "Ồ", nil, 7891, nil],
      7891 => [0, 0, "ồ", "ồ", 7890, nil, 7890],
      7892 => [0, 0, "Ổ", "Ổ", nil, 7893, nil],
      7893 => [0, 0, "ổ", "ổ", 7892, nil, 7892],
      7894 => [0, 0, "Ỗ", "Ỗ", nil, 7895, nil],
      7895 => [0, 0, "ỗ", "ỗ", 7894, nil, 7894],
      7896 => [0, 0, "Ộ", "Ộ", nil, 7897, nil],
      7897 => [0, 0, "ộ", "ộ", 7896, nil, 7896],
      7898 => [0, 0, "Ớ", "Ớ", nil, 7899, nil],
      7899 => [0, 0, "ớ", "ớ", 7898, nil, 7898],
      7900 => [0, 0, "Ờ", "Ờ", nil, 7901, nil],
      7901 => [0, 0, "ờ", "ờ", 7900, nil, 7900],
      7902 => [0, 0, "Ở", "Ở", nil, 7903, nil],
      7903 => [0, 0, "ở", "ở", 7902, nil, 7902],
      7904 => [0, 0, "Ỡ", "Ỡ", nil, 7905, nil],
      7905 => [0, 0, "ỡ", "ỡ", 7904, nil, 7904],
      7906 => [0, 0, "Ợ", "Ợ", nil, 7907, nil],
      7907 => [0, 0, "ợ", "ợ", 7906, nil, 7906],
      7908 => [0, 0, "Ụ", "Ụ", nil, 7909, nil],
      7909 => [0, 0, "ụ", "ụ", 7908, nil, 7908],
      7910 => [0, 0, "Ủ", "Ủ", nil, 7911, nil],
      7911 => [0, 0, "ủ", "ủ", 7910, nil, 7910],
      7912 => [0, 0, "Ứ", "Ứ", nil, 7913, nil],
      7913 => [0, 0, "ứ", "ứ", 7912, nil, 7912],
      7914 => [0, 0, "Ừ", "Ừ", nil, 7915, nil],
      7915 => [0, 0, "ừ", "ừ", 7914, nil, 7914],
      7916 => [0, 0, "Ử", "Ử", nil, 7917, nil],
      7917 => [0, 0, "ử", "ử", 7916, nil, 7916],
      7918 => [0, 0, "Ữ", "Ữ", nil, 7919, nil],
      7919 => [0, 0, "ữ", "ữ", 7918, nil, 7918],
      7920 => [0, 0, "Ự", "Ự", nil, 7921, nil],
      7921 => [0, 0, "ự", "ự", 7920, nil, 7920],
      7922 => [0, 0, "Ỳ", "Ỳ", nil, 7923, nil],
      7923 => [0, 0, "ỳ", "ỳ", 7922, nil, 7922],
      7924 => [0, 0, "Ỵ", "Ỵ", nil, 7925, nil],
      7925 => [0, 0, "ỵ", "ỵ", 7924, nil, 7924],
      7926 => [0, 0, "Ỷ", "Ỷ", nil, 7927, nil],
      7927 => [0, 0, "ỷ", "ỷ", 7926, nil, 7926],
      7928 => [0, 0, "Ỹ", "Ỹ", nil, 7929, nil],
      7929 => [0, 0, "ỹ", "ỹ", 7928, nil, 7928],
      7936 => [0, 0, "ἀ", "ἀ", 7944, nil, 7944],
      7937 => [0, 0, "ἁ", "ἁ", 7945, nil, 7945],
      7938 => [0, 0, "ἂ", "ἂ", 7946, nil, 7946],
      7939 => [0, 0, "ἃ", "ἃ", 7947, nil, 7947],
      7940 => [0, 0, "ἄ", "ἄ", 7948, nil, 7948],
      7941 => [0, 0, "ἅ", "ἅ", 7949, nil, 7949],
      7942 => [0, 0, "ἆ", "ἆ", 7950, nil, 7950],
      7943 => [0, 0, "ἇ", "ἇ", 7951, nil, 7951],
      7944 => [0, 0, "Ἀ", "Ἀ", nil, 7936, nil],
      7945 => [0, 0, "Ἁ", "Ἁ", nil, 7937, nil],
      7946 => [0, 0, "Ἂ", "Ἂ", nil, 7938, nil],
      7947 => [0, 0, "Ἃ", "Ἃ", nil, 7939, nil],
      7948 => [0, 0, "Ἄ", "Ἄ", nil, 7940, nil],
      7949 => [0, 0, "Ἅ", "Ἅ", nil, 7941, nil],
      7950 => [0, 0, "Ἆ", "Ἆ", nil, 7942, nil],
      7951 => [0, 0, "Ἇ", "Ἇ", nil, 7943, nil],
      7952 => [0, 0, "ἐ", "ἐ", 7960, nil, 7960],
      7953 => [0, 0, "ἑ", "ἑ", 7961, nil, 7961],
      7954 => [0, 0, "ἒ", "ἒ", 7962, nil, 7962],
      7955 => [0, 0, "ἓ", "ἓ", 7963, nil, 7963],
      7956 => [0, 0, "ἔ", "ἔ", 7964, nil, 7964],
      7957 => [0, 0, "ἕ", "ἕ", 7965, nil, 7965],
      7960 => [0, 0, "Ἐ", "Ἐ", nil, 7952, nil],
      7961 => [0, 0, "Ἑ", "Ἑ", nil, 7953, nil],
      7962 => [0, 0, "Ἒ", "Ἒ", nil, 7954, nil],
      7963 => [0, 0, "Ἓ", "Ἓ", nil, 7955, nil],
      7964 => [0, 0, "Ἔ", "Ἔ", nil, 7956, nil],
      7965 => [0, 0, "Ἕ", "Ἕ", nil, 7957, nil],
      7968 => [0, 0, "ἠ", "ἠ", 7976, nil, 7976],
      7969 => [0, 0, "ἡ", "ἡ", 7977, nil, 7977],
      7970 => [0, 0, "ἢ", "ἢ", 7978, nil, 7978],
      7971 => [0, 0, "ἣ", "ἣ", 7979, nil, 7979],
      7972 => [0, 0, "ἤ", "ἤ", 7980, nil, 7980],
      7973 => [0, 0, "ἥ", "ἥ", 7981, nil, 7981],
      7974 => [0, 0, "ἦ", "ἦ", 7982, nil, 7982],
      7975 => [0, 0, "ἧ", "ἧ", 7983, nil, 7983],
      7976 => [0, 0, "Ἠ", "Ἠ", nil, 7968, nil],
      7977 => [0, 0, "Ἡ", "Ἡ", nil, 7969, nil],
      7978 => [0, 0, "Ἢ", "Ἢ", nil, 7970, nil],
      7979 => [0, 0, "Ἣ", "Ἣ", nil, 7971, nil],
      7980 => [0, 0, "Ἤ", "Ἤ", nil, 7972, nil],
      7981 => [0, 0, "Ἥ", "Ἥ", nil, 7973, nil],
      7982 => [0, 0, "Ἦ", "Ἦ", nil, 7974, nil],
      7983 => [0, 0, "Ἧ", "Ἧ", nil, 7975, nil],
      7984 => [0, 0, "ἰ", "ἰ", 7992, nil, 7992],
      7985 => [0, 0, "ἱ", "ἱ", 7993, nil, 7993],
      7986 => [0, 0, "ἲ", "ἲ", 7994, nil, 7994],
      7987 => [0, 0, "ἳ", "ἳ", 7995, nil, 7995],
      7988 => [0, 0, "ἴ", "ἴ", 7996, nil, 7996],
      7989 => [0, 0, "ἵ", "ἵ", 7997, nil, 7997],
      7990 => [0, 0, "ἶ", "ἶ", 7998, nil, 7998],
      7991 => [0, 0, "ἷ", "ἷ", 7999, nil, 7999],
      7992 => [0, 0, "Ἰ", "Ἰ", nil, 7984, nil],
      7993 => [0, 0, "Ἱ", "Ἱ", nil, 7985, nil],
      7994 => [0, 0, "Ἲ", "Ἲ", nil, 7986, nil],
      7995 => [0, 0, "Ἳ", "Ἳ", nil, 7987, nil],
      7996 => [0, 0, "Ἴ", "Ἴ", nil, 7988, nil],
      7997 => [0, 0, "Ἵ", "Ἵ", nil, 7989, nil],
      7998 => [0, 0, "Ἶ", "Ἶ", nil, 7990, nil],
      7999 => [0, 0, "Ἷ", "Ἷ", nil, 7991, nil],
      8000 => [0, 0, "ὀ", "ὀ", 8008, nil, 8008],
      8001 => [0, 0, "ὁ", "ὁ", 8009, nil, 8009],
      8002 => [0, 0, "ὂ", "ὂ", 8010, nil, 8010],
      8003 => [0, 0, "ὃ", "ὃ", 8011, nil, 8011],
      8004 => [0, 0, "ὄ", "ὄ", 8012, nil, 8012],
      8005 => [0, 0, "ὅ", "ὅ", 8013, nil, 8013],
      8008 => [0, 0, "Ὀ", "Ὀ", nil, 8000, nil],
      8009 => [0, 0, "Ὁ", "Ὁ", nil, 8001, nil],
      8010 => [0, 0, "Ὂ", "Ὂ", nil, 8002, nil],
      8011 => [0, 0, "Ὃ", "Ὃ", nil, 8003, nil],
      8012 => [0, 0, "Ὄ", "Ὄ", nil, 8004, nil],
      8013 => [0, 0, "Ὅ", "Ὅ", nil, 8005, nil],
      8016 => [0, 0, "ὐ", "ὐ", nil, nil, nil],
      8017 => [0, 0, "ὑ", "ὑ", 8025, nil, 8025],
      8018 => [0, 0, "ὒ", "ὒ", nil, nil, nil],
      8019 => [0, 0, "ὓ", "ὓ", 8027, nil, 8027],
      8020 => [0, 0, "ὔ", "ὔ", nil, nil, nil],
      8021 => [0, 0, "ὕ", "ὕ", 8029, nil, 8029],
      8022 => [0, 0, "ὖ", "ὖ", nil, nil, nil],
      8023 => [0, 0, "ὗ", "ὗ", 8031, nil, 8031],
      8025 => [0, 0, "Ὑ", "Ὑ", nil, 8017, nil],
      8027 => [0, 0, "Ὓ", "Ὓ", nil, 8019, nil],
      8029 => [0, 0, "Ὕ", "Ὕ", nil, 8021, nil],
      8031 => [0, 0, "Ὗ", "Ὗ", nil, 8023, nil],
      8032 => [0, 0, "ὠ", "ὠ", 8040, nil, 8040],
      8033 => [0, 0, "ὡ", "ὡ", 8041, nil, 8041],
      8034 => [0, 0, "ὢ", "ὢ", 8042, nil, 8042],
      8035 => [0, 0, "ὣ", "ὣ", 8043, nil, 8043],
      8036 => [0, 0, "ὤ", "ὤ", 8044, nil, 8044],
      8037 => [0, 0, "ὥ", "ὥ", 8045, nil, 8045],
      8038 => [0, 0, "ὦ", "ὦ", 8046, nil, 8046],
      8039 => [0, 0, "ὧ", "ὧ", 8047, nil, 8047],
      8040 => [0, 0, "Ὠ", "Ὠ", nil, 8032, nil],
      8041 => [0, 0, "Ὡ", "Ὡ", nil, 8033, nil],
      8042 => [0, 0, "Ὢ", "Ὢ", nil, 8034, nil],
      8043 => [0, 0, "Ὣ", "Ὣ", nil, 8035, nil],
      8044 => [0, 0, "Ὤ", "Ὤ", nil, 8036, nil],
      8045 => [0, 0, "Ὥ", "Ὥ", nil, 8037, nil],
      8046 => [0, 0, "Ὦ", "Ὦ", nil, 8038, nil],
      8047 => [0, 0, "Ὧ", "Ὧ", nil, 8039, nil],
      8048 => [0, 0, "ὰ", "ὰ", 8122, nil, 8122],
      8049 => [0, 2, "ά", "ά", 8123, nil, 8123],
      8050 => [0, 0, "ὲ", "ὲ", 8136, nil, 8136],
      8051 => [0, 2, "έ", "έ", 8137, nil, 8137],
      8052 => [0, 0, "ὴ", "ὴ", 8138, nil, 8138],
      8053 => [0, 2, "ή", "ή", 8139, nil, 8139],
      8054 => [0, 0, "ὶ", "ὶ", 8154, nil, 8154],
      8055 => [0, 2, "ί", "ί", 8155, nil, 8155],
      8056 => [0, 0, "ὸ", "ὸ", 8184, nil, 8184],
      8057 => [0, 2, "ό", "ό", 8185, nil, 8185],
      8058 => [0, 0, "ὺ", "ὺ", 8170, nil, 8170],
      8059 => [0, 2, "ύ", "ύ", 8171, nil, 8171],
      8060 => [0, 0, "ὼ", "ὼ", 8186, nil, 8186],
      8061 => [0, 2, "ώ", "ώ", 8187, nil, 8187],
      8064 => [0, 0, "ᾀ", "ᾀ", 8072, nil, 8072],
      8065 => [0, 0, "ᾁ", "ᾁ", 8073, nil, 8073],
      8066 => [0, 0, "ᾂ", "ᾂ", 8074, nil, 8074],
      8067 => [0, 0, "ᾃ", "ᾃ", 8075, nil, 8075],
      8068 => [0, 0, "ᾄ", "ᾄ", 8076, nil, 8076],
      8069 => [0, 0, "ᾅ", "ᾅ", 8077, nil, 8077],
      8070 => [0, 0, "ᾆ", "ᾆ", 8078, nil, 8078],
      8071 => [0, 0, "ᾇ", "ᾇ", 8079, nil, 8079],
      8072 => [0, 0, "ᾈ", "ᾈ", nil, 8064, nil],
      8073 => [0, 0, "ᾉ", "ᾉ", nil, 8065, nil],
      8074 => [0, 0, "ᾊ", "ᾊ", nil, 8066, nil],
      8075 => [0, 0, "ᾋ", "ᾋ", nil, 8067, nil],
      8076 => [0, 0, "ᾌ", "ᾌ", nil, 8068, nil],
      8077 => [0, 0, "ᾍ", "ᾍ", nil, 8069, nil],
      8078 => [0, 0, "ᾎ", "ᾎ", nil, 8070, nil],
      8079 => [0, 0, "ᾏ", "ᾏ", nil, 8071, nil],
      8080 => [0, 0, "ᾐ", "ᾐ", 8088, nil, 8088],
      8081 => [0, 0, "ᾑ", "ᾑ", 8089, nil, 8089],
      8082 => [0, 0, "ᾒ", "ᾒ", 8090, nil, 8090],
      8083 => [0, 0, "ᾓ", "ᾓ", 8091, nil, 8091],
      8084 => [0, 0, "ᾔ", "ᾔ", 8092, nil, 8092],
      8085 => [0, 0, "ᾕ", "ᾕ", 8093, nil, 8093],
      8086 => [0, 0, "ᾖ", "ᾖ", 8094, nil, 8094],
      8087 => [0, 0, "ᾗ", "ᾗ", 8095, nil, 8095],
      8088 => [0, 0, "ᾘ", "ᾘ", nil, 8080, nil],
      8089 => [0, 0, "ᾙ", "ᾙ", nil, 8081, nil],
      8090 => [0, 0, "ᾚ", "ᾚ", nil, 8082, nil],
      8091 => [0, 0, "ᾛ", "ᾛ", nil, 8083, nil],
      8092 => [0, 0, "ᾜ", "ᾜ", nil, 8084, nil],
      8093 => [0, 0, "ᾝ", "ᾝ", nil, 8085, nil],
      8094 => [0, 0, "ᾞ", "ᾞ", nil, 8086, nil],
      8095 => [0, 0, "ᾟ", "ᾟ", nil, 8087, nil],
      8096 => [0, 0, "ᾠ", "ᾠ", 8104, nil, 8104],
      8097 => [0, 0, "ᾡ", "ᾡ", 8105, nil, 8105],
      8098 => [0, 0, "ᾢ", "ᾢ", 8106, nil, 8106],
      8099 => [0, 0, "ᾣ", "ᾣ", 8107, nil, 8107],
      8100 => [0, 0, "ᾤ", "ᾤ", 8108, nil, 8108],
      8101 => [0, 0, "ᾥ", "ᾥ", 8109, nil, 8109],
      8102 => [0, 0, "ᾦ", "ᾦ", 8110, nil, 8110],
      8103 => [0, 0, "ᾧ", "ᾧ", 8111, nil, 8111],
      8104 => [0, 0, "ᾨ", "ᾨ", nil, 8096, nil],
      8105 => [0, 0, "ᾩ", "ᾩ", nil, 8097, nil],
      8106 => [0, 0, "ᾪ", "ᾪ", nil, 8098, nil],
      8107 => [0, 0, "ᾫ", "ᾫ", nil, 8099, nil],
      8108 => [0, 0, "ᾬ", "ᾬ", nil, 8100, nil],
      8109 => [0, 0, "ᾭ", "ᾭ", nil, 8101, nil],
      8110 => [0, 0, "ᾮ", "ᾮ", nil, 8102, nil],
      8111 => [0, 0, "ᾯ", "ᾯ", nil, 8103, nil],
      8112 => [0, 0, "ᾰ", "ᾰ", 8120, nil, 8120],
      8113 => [0, 0, "ᾱ", "ᾱ", 8121, nil, 8121],
      8114 => [0, 0, "ᾲ", "ᾲ", nil, nil, nil],
      8115 => [0, 0, "ᾳ", "ᾳ", 8124, nil, 8124],
      8116 => [0, 0, "ᾴ", "ᾴ", nil, nil, nil],
      8118 => [0, 0, "ᾶ", "ᾶ", nil, nil, nil],
      8119 => [0, 0, "ᾷ", "ᾷ", nil, nil, nil],
      8120 => [0, 0, "Ᾰ", "Ᾰ", nil, 8112, nil],
      8121 => [0, 0, "Ᾱ", "Ᾱ", nil, 8113, nil],
      8122 => [0, 0, "Ὰ", "Ὰ", nil, 8048, nil],
      8123 => [0, 2, "Ά", "Ά", nil, 8049, nil],
      8124 => [0, 0, "ᾼ", "ᾼ", nil, 8115, nil],
      8125 => [0, 0, nil, " ̓", nil, nil, nil],
      8126 => [0, 2, "ι", "ι", 921, nil, 921],
      8127 => [0, 0, nil, " ̓", nil, nil, nil],
      8128 => [0, 0, nil, " ͂", nil, nil, nil],
      8129 => [0, 0, "῁", "῁", nil, nil, nil],
      8130 => [0, 0, "ῂ", "ῂ", nil, nil, nil],
      8131 => [0, 0, "ῃ", "ῃ", 8140, nil, 8140],
      8132 => [0, 0, "ῄ", "ῄ", nil, nil, nil],
      8134 => [0, 0, "ῆ", "ῆ", nil, nil, nil],
      8135 => [0, 0, "ῇ", "ῇ", nil, nil, nil],
      8136 => [0, 0, "Ὲ", "Ὲ", nil, 8050, nil],
      8137 => [0, 2, "Έ", "Έ", nil, 8051, nil],
      8138 => [0, 0, "Ὴ", "Ὴ", nil, 8052, nil],
      8139 => [0, 2, "Ή", "Ή", nil, 8053, nil],
      8140 => [0, 0, "ῌ", "ῌ", nil, 8131, nil],
      8141 => [0, 0, "῍", "῍", nil, nil, nil],
      8142 => [0, 0, "῎", "῎", nil, nil, nil],
      8143 => [0, 0, "῏", "῏", nil, nil, nil],
      8144 => [0, 0, "ῐ", "ῐ", 8152, nil, 8152],
      8145 => [0, 0, "ῑ", "ῑ", 8153, nil, 8153],
      8146 => [0, 0, "ῒ", "ῒ", nil, nil, nil],
      8147 => [0, 2, "ΐ", "ΐ", nil, nil, nil],
      8150 => [0, 0, "ῖ", "ῖ", nil, nil, nil],
      8151 => [0, 0, "ῗ", "ῗ", nil, nil, nil],
      8152 => [0, 0, "Ῐ", "Ῐ", nil, 8144, nil],
      8153 => [0, 0, "Ῑ", "Ῑ", nil, 8145, nil],
      8154 => [0, 0, "Ὶ", "Ὶ", nil, 8054, nil],
      8155 => [0, 2, "Ί", "Ί", nil, 8055, nil],
      8157 => [0, 0, "῝", "῝", nil, nil, nil],
      8158 => [0, 0, "῞", "῞", nil, nil, nil],
      8159 => [0, 0, "῟", "῟", nil, nil, nil],
      8160 => [0, 0, "ῠ", "ῠ", 8168, nil, 8168],
      8161 => [0, 0, "ῡ", "ῡ", 8169, nil, 8169],
      8162 => [0, 0, "ῢ", "ῢ", nil, nil, nil],
      8163 => [0, 2, "ΰ", "ΰ", nil, nil, nil],
      8164 => [0, 0, "ῤ", "ῤ", nil, nil, nil],
      8165 => [0, 0, "ῥ", "ῥ", 8172, nil, 8172],
      8166 => [0, 0, "ῦ", "ῦ", nil, nil, nil],
      8167 => [0, 0, "ῧ", "ῧ", nil, nil, nil],
      8168 => [0, 0, "Ῠ", "Ῠ", nil, 8160, nil],
      8169 => [0, 0, "Ῡ", "Ῡ", nil, 8161, nil],
      8170 => [0, 0, "Ὺ", "Ὺ", nil, 8058, nil],
      8171 => [0, 2, "Ύ", "Ύ", nil, 8059, nil],
      8172 => [0, 0, "Ῥ", "Ῥ", nil, 8165, nil],
      8173 => [0, 0, "῭", "῭", nil, nil, nil],
      8174 => [0, 2, "΅", "΅", nil, nil, nil],
      8175 => [0, 2, "`", "`", nil, nil, nil],
      8178 => [0, 0, "ῲ", "ῲ", nil, nil, nil],
      8179 => [0, 0, "ῳ", "ῳ", 8188, nil, 8188],
      8180 => [0, 0, "ῴ", "ῴ", nil, nil, nil],
      8182 => [0, 0, "ῶ", "ῶ", nil, nil, nil],
      8183 => [0, 0, "ῷ", "ῷ", nil, nil, nil],
      8184 => [0, 0, "Ὸ", "Ὸ", nil, 8056, nil],
      8185 => [0, 2, "Ό", "Ό", nil, 8057, nil],
      8186 => [0, 0, "Ὼ", "Ὼ", nil, 8060, nil],
      8187 => [0, 2, "Ώ", "Ώ", nil, 8061, nil],
      8188 => [0, 0, "ῼ", "ῼ", nil, 8179, nil],
      8189 => [0, 2, "´", "´", nil, nil, nil],
      8190 => [0, 0, nil, " ̔", nil, nil, nil],
      8192 => [0, 2, " ", " ", nil, nil, nil],
      8193 => [0, 2, " ", " ", nil, nil, nil],
      8194 => [0, 0, nil, " ", nil, nil, nil],
      8195 => [0, 0, nil, " ", nil, nil, nil],
      8196 => [0, 0, nil, " ", nil, nil, nil],
      8197 => [0, 0, nil, " ", nil, nil, nil],
      8198 => [0, 0, nil, " ", nil, nil, nil],
      8199 => [0, 0, nil, " ", nil, nil, nil],
      8200 => [0, 0, nil, " ", nil, nil, nil],
      8201 => [0, 0, nil, " ", nil, nil, nil],
      8202 => [0, 0, nil, " ", nil, nil, nil],
      8209 => [0, 0, nil, "‐", nil, nil, nil],
      8215 => [0, 0, nil, " ̳", nil, nil, nil],
      8228 => [0, 0, nil, ".", nil, nil, nil],
      8229 => [0, 0, nil, "..", nil, nil, nil],
      8230 => [0, 0, nil, "...", nil, nil, nil],
      8239 => [0, 0, nil, " ", nil, nil, nil],
      8243 => [0, 0, nil, "′′", nil, nil, nil],
      8244 => [0, 0, nil, "′′′", nil, nil, nil],
      8246 => [0, 0, nil, "‵‵", nil, nil, nil],
      8247 => [0, 0, nil, "‵‵‵", nil, nil, nil],
      8252 => [0, 0, nil, "!!", nil, nil, nil],
      8254 => [0, 0, nil, " ̅", nil, nil, nil],
      8264 => [0, 0, nil, "?!", nil, nil, nil],
      8265 => [0, 0, nil, "!?", nil, nil, nil],
      8304 => [0, 0, nil, "0", nil, nil, nil],
      8308 => [0, 0, nil, "4", nil, nil, nil],
      8309 => [0, 0, nil, "5", nil, nil, nil],
      8310 => [0, 0, nil, "6", nil, nil, nil],
      8311 => [0, 0, nil, "7", nil, nil, nil],
      8312 => [0, 0, nil, "8", nil, nil, nil],
      8313 => [0, 0, nil, "9", nil, nil, nil],
      8314 => [0, 0, nil, "+", nil, nil, nil],
      8315 => [0, 0, nil, "−", nil, nil, nil],
      8316 => [0, 0, nil, "=", nil, nil, nil],
      8317 => [0, 0, nil, "(", nil, nil, nil],
      8318 => [0, 0, nil, ")", nil, nil, nil],
      8319 => [0, 0, nil, "n", nil, nil, nil],
      8320 => [0, 0, nil, "0", nil, nil, nil],
      8321 => [0, 0, nil, "1", nil, nil, nil],
      8322 => [0, 0, nil, "2", nil, nil, nil],
      8323 => [0, 0, nil, "3", nil, nil, nil],
      8324 => [0, 0, nil, "4", nil, nil, nil],
      8325 => [0, 0, nil, "5", nil, nil, nil],
      8326 => [0, 0, nil, "6", nil, nil, nil],
      8327 => [0, 0, nil, "7", nil, nil, nil],
      8328 => [0, 0, nil, "8", nil, nil, nil],
      8329 => [0, 0, nil, "9", nil, nil, nil],
      8330 => [0, 0, nil, "+", nil, nil, nil],
      8331 => [0, 0, nil, "−", nil, nil, nil],
      8332 => [0, 0, nil, "=", nil, nil, nil],
      8333 => [0, 0, nil, "(", nil, nil, nil],
      8334 => [0, 0, nil, ")", nil, nil, nil],
      8360 => [0, 0, nil, "Rs", nil, nil, nil],
      8400 => [230, 0, nil, nil, nil, nil, nil],
      8401 => [230, 0, nil, nil, nil, nil, nil],
      8402 => [1, 0, nil, nil, nil, nil, nil],
      8403 => [1, 0, nil, nil, nil, nil, nil],
      8404 => [230, 0, nil, nil, nil, nil, nil],
      8405 => [230, 0, nil, nil, nil, nil, nil],
      8406 => [230, 0, nil, nil, nil, nil, nil],
      8407 => [230, 0, nil, nil, nil, nil, nil],
      8408 => [1, 0, nil, nil, nil, nil, nil],
      8409 => [1, 0, nil, nil, nil, nil, nil],
      8410 => [1, 0, nil, nil, nil, nil, nil],
      8411 => [230, 0, nil, nil, nil, nil, nil],
      8412 => [230, 0, nil, nil, nil, nil, nil],
      8417 => [230, 0, nil, nil, nil, nil, nil],
      8448 => [0, 0, nil, "a/c", nil, nil, nil],
      8449 => [0, 0, nil, "a/s", nil, nil, nil],
      8450 => [0, 0, nil, "C", nil, nil, nil],
      8451 => [0, 0, nil, "°C", nil, nil, nil],
      8453 => [0, 0, nil, "c/o", nil, nil, nil],
      8454 => [0, 0, nil, "c/u", nil, nil, nil],
      8455 => [0, 0, nil, "Ɛ", nil, nil, nil],
      8457 => [0, 0, nil, "°F", nil, nil, nil],
      8458 => [0, 0, nil, "g", nil, nil, nil],
      8459 => [0, 0, nil, "H", nil, nil, nil],
      8460 => [0, 0, nil, "H", nil, nil, nil],
      8461 => [0, 0, nil, "H", nil, nil, nil],
      8462 => [0, 0, nil, "h", nil, nil, nil],
      8463 => [0, 0, nil, "ħ", nil, nil, nil],
      8464 => [0, 0, nil, "I", nil, nil, nil],
      8465 => [0, 0, nil, "I", nil, nil, nil],
      8466 => [0, 0, nil, "L", nil, nil, nil],
      8467 => [0, 0, nil, "l", nil, nil, nil],
      8469 => [0, 0, nil, "N", nil, nil, nil],
      8470 => [0, 0, nil, "No", nil, nil, nil],
      8473 => [0, 0, nil, "P", nil, nil, nil],
      8474 => [0, 0, nil, "Q", nil, nil, nil],
      8475 => [0, 0, nil, "R", nil, nil, nil],
      8476 => [0, 0, nil, "R", nil, nil, nil],
      8477 => [0, 0, nil, "R", nil, nil, nil],
      8480 => [0, 0, nil, "SM", nil, nil, nil],
      8481 => [0, 0, nil, "TEL", nil, nil, nil],
      8482 => [0, 0, nil, "TM", nil, nil, nil],
      8484 => [0, 0, nil, "Z", nil, nil, nil],
      8486 => [0, 2, "Ω", "Ω", nil, 969, nil],
      8488 => [0, 0, nil, "Z", nil, nil, nil],
      8490 => [0, 2, "K", "K", nil, 107, nil],
      8491 => [0, 2, "Å", "Å", nil, 229, nil],
      8492 => [0, 0, nil, "B", nil, nil, nil],
      8493 => [0, 0, nil, "C", nil, nil, nil],
      8495 => [0, 0, nil, "e", nil, nil, nil],
      8496 => [0, 0, nil, "E", nil, nil, nil],
      8497 => [0, 0, nil, "F", nil, nil, nil],
      8499 => [0, 0, nil, "M", nil, nil, nil],
      8500 => [0, 0, nil, "o", nil, nil, nil],
      8501 => [0, 0, nil, "א", nil, nil, nil],
      8502 => [0, 0, nil, "ב", nil, nil, nil],
      8503 => [0, 0, nil, "ג", nil, nil, nil],
      8504 => [0, 0, nil, "ד", nil, nil, nil],
      8505 => [0, 0, nil, "i", nil, nil, nil],
      8531 => [0, 0, nil, "1⁄3", nil, nil, nil],
      8532 => [0, 0, nil, "2⁄3", nil, nil, nil],
      8533 => [0, 0, nil, "1⁄5", nil, nil, nil],
      8534 => [0, 0, nil, "2⁄5", nil, nil, nil],
      8535 => [0, 0, nil, "3⁄5", nil, nil, nil],
      8536 => [0, 0, nil, "4⁄5", nil, nil, nil],
      8537 => [0, 0, nil, "1⁄6", nil, nil, nil],
      8538 => [0, 0, nil, "5⁄6", nil, nil, nil],
      8539 => [0, 0, nil, "1⁄8", nil, nil, nil],
      8540 => [0, 0, nil, "3⁄8", nil, nil, nil],
      8541 => [0, 0, nil, "5⁄8", nil, nil, nil],
      8542 => [0, 0, nil, "7⁄8", nil, nil, nil],
      8543 => [0, 0, nil, "1⁄", nil, nil, nil],
      8544 => [0, 0, nil, "I", nil, 8560, nil],
      8545 => [0, 0, nil, "II", nil, 8561, nil],
      8546 => [0, 0, nil, "III", nil, 8562, nil],
      8547 => [0, 0, nil, "IV", nil, 8563, nil],
      8548 => [0, 0, nil, "V", nil, 8564, nil],
      8549 => [0, 0, nil, "VI", nil, 8565, nil],
      8550 => [0, 0, nil, "VII", nil, 8566, nil],
      8551 => [0, 0, nil, "VIII", nil, 8567, nil],
      8552 => [0, 0, nil, "IX", nil, 8568, nil],
      8553 => [0, 0, nil, "X", nil, 8569, nil],
      8554 => [0, 0, nil, "XI", nil, 8570, nil],
      8555 => [0, 0, nil, "XII", nil, 8571, nil],
      8556 => [0, 0, nil, "L", nil, 8572, nil],
      8557 => [0, 0, nil, "C", nil, 8573, nil],
      8558 => [0, 0, nil, "D", nil, 8574, nil],
      8559 => [0, 0, nil, "M", nil, 8575, nil],
      8560 => [0, 0, nil, "i", 8544, nil, 8544],
      8561 => [0, 0, nil, "ii", 8545, nil, 8545],
      8562 => [0, 0, nil, "iii", 8546, nil, 8546],
      8563 => [0, 0, nil, "iv", 8547, nil, 8547],
      8564 => [0, 0, nil, "v", 8548, nil, 8548],
      8565 => [0, 0, nil, "vi", 8549, nil, 8549],
      8566 => [0, 0, nil, "vii", 8550, nil, 8550],
      8567 => [0, 0, nil, "viii", 8551, nil, 8551],
      8568 => [0, 0, nil, "ix", 8552, nil, 8552],
      8569 => [0, 0, nil, "x", 8553, nil, 8553],
      8570 => [0, 0, nil, "xi", 8554, nil, 8554],
      8571 => [0, 0, nil, "xii", 8555, nil, 8555],
      8572 => [0, 0, nil, "l", 8556, nil, 8556],
      8573 => [0, 0, nil, "c", 8557, nil, 8557],
      8574 => [0, 0, nil, "d", 8558, nil, 8558],
      8575 => [0, 0, nil, "m", 8559, nil, 8559],
      8602 => [0, 0, "↚", "↚", nil, nil, nil],
      8603 => [0, 0, "↛", "↛", nil, nil, nil],
      8622 => [0, 0, "↮", "↮", nil, nil, nil],
      8653 => [0, 0, "⇍", "⇍", nil, nil, nil],
      8654 => [0, 0, "⇎", "⇎", nil, nil, nil],
      8655 => [0, 0, "⇏", "⇏", nil, nil, nil],
      8708 => [0, 0, "∄", "∄", nil, nil, nil],
      8713 => [0, 0, "∉", "∉", nil, nil, nil],
      8716 => [0, 0, "∌", "∌", nil, nil, nil],
      8740 => [0, 0, "∤", "∤", nil, nil, nil],
      8742 => [0, 0, "∦", "∦", nil, nil, nil],
      8748 => [0, 0, nil, "∫∫", nil, nil, nil],
      8749 => [0, 0, nil, "∫∫∫", nil, nil, nil],
      8751 => [0, 0, nil, "∮∮", nil, nil, nil],
      8752 => [0, 0, nil, "∮∮∮", nil, nil, nil],
      8769 => [0, 0, "≁", "≁", nil, nil, nil],
      8772 => [0, 0, "≄", "≄", nil, nil, nil],
      8775 => [0, 0, "≇", "≇", nil, nil, nil],
      8777 => [0, 0, "≉", "≉", nil, nil, nil],
      8800 => [0, 0, "≠", "≠", nil, nil, nil],
      8802 => [0, 0, "≢", "≢", nil, nil, nil],
      8813 => [0, 0, "≭", "≭", nil, nil, nil],
      8814 => [0, 0, "≮", "≮", nil, nil, nil],
      8815 => [0, 0, "≯", "≯", nil, nil, nil],
      8816 => [0, 0, "≰", "≰", nil, nil, nil],
      8817 => [0, 0, "≱", "≱", nil, nil, nil],
      8820 => [0, 0, "≴", "≴", nil, nil, nil],
      8821 => [0, 0, "≵", "≵", nil, nil, nil],
      8824 => [0, 0, "≸", "≸", nil, nil, nil],
      8825 => [0, 0, "≹", "≹", nil, nil, nil],
      8832 => [0, 0, "⊀", "⊀", nil, nil, nil],
      8833 => [0, 0, "⊁", "⊁", nil, nil, nil],
      8836 => [0, 0, "⊄", "⊄", nil, nil, nil],
      8837 => [0, 0, "⊅", "⊅", nil, nil, nil],
      8840 => [0, 0, "⊈", "⊈", nil, nil, nil],
      8841 => [0, 0, "⊉", "⊉", nil, nil, nil],
      8876 => [0, 0, "⊬", "⊬", nil, nil, nil],
      8877 => [0, 0, "⊭", "⊭", nil, nil, nil],
      8878 => [0, 0, "⊮", "⊮", nil, nil, nil],
      8879 => [0, 0, "⊯", "⊯", nil, nil, nil],
      8928 => [0, 0, "⋠", "⋠", nil, nil, nil],
      8929 => [0, 0, "⋡", "⋡", nil, nil, nil],
      8930 => [0, 0, "⋢", "⋢", nil, nil, nil],
      8931 => [0, 0, "⋣", "⋣", nil, nil, nil],
      8938 => [0, 0, "⋪", "⋪", nil, nil, nil],
      8939 => [0, 0, "⋫", "⋫", nil, nil, nil],
      8940 => [0, 0, "⋬", "⋬", nil, nil, nil],
      8941 => [0, 0, "⋭", "⋭", nil, nil, nil],
      9001 => [0, 2, "〈", "〈", nil, nil, nil],
      9002 => [0, 2, "〉", "〉", nil, nil, nil],
      9312 => [0, 0, nil, "1", nil, nil, nil],
      9313 => [0, 0, nil, "2", nil, nil, nil],
      9314 => [0, 0, nil, "3", nil, nil, nil],
      9315 => [0, 0, nil, "4", nil, nil, nil],
      9316 => [0, 0, nil, "5", nil, nil, nil],
      9317 => [0, 0, nil, "6", nil, nil, nil],
      9318 => [0, 0, nil, "7", nil, nil, nil],
      9319 => [0, 0, nil, "8", nil, nil, nil],
      9320 => [0, 0, nil, "9", nil, nil, nil],
      9321 => [0, 0, nil, "10", nil, nil, nil],
      9322 => [0, 0, nil, "11", nil, nil, nil],
      9323 => [0, 0, nil, "12", nil, nil, nil],
      9324 => [0, 0, nil, "13", nil, nil, nil],
      9325 => [0, 0, nil, "14", nil, nil, nil],
      9326 => [0, 0, nil, "15", nil, nil, nil],
      9327 => [0, 0, nil, "16", nil, nil, nil],
      9328 => [0, 0, nil, "17", nil, nil, nil],
      9329 => [0, 0, nil, "18", nil, nil, nil],
      9330 => [0, 0, nil, "19", nil, nil, nil],
      9331 => [0, 0, nil, "20", nil, nil, nil],
      9332 => [0, 0, nil, "(1)", nil, nil, nil],
      9333 => [0, 0, nil, "(2)", nil, nil, nil],
      9334 => [0, 0, nil, "(3)", nil, nil, nil],
      9335 => [0, 0, nil, "(4)", nil, nil, nil],
      9336 => [0, 0, nil, "(5)", nil, nil, nil],
      9337 => [0, 0, nil, "(6)", nil, nil, nil],
      9338 => [0, 0, nil, "(7)", nil, nil, nil],
      9339 => [0, 0, nil, "(8)", nil, nil, nil],
      9340 => [0, 0, nil, "(9)", nil, nil, nil],
      9341 => [0, 0, nil, "(10)", nil, nil, nil],
      9342 => [0, 0, nil, "(11)", nil, nil, nil],
      9343 => [0, 0, nil, "(12)", nil, nil, nil],
      9344 => [0, 0, nil, "(13)", nil, nil, nil],
      9345 => [0, 0, nil, "(14)", nil, nil, nil],
      9346 => [0, 0, nil, "(15)", nil, nil, nil],
      9347 => [0, 0, nil, "(16)", nil, nil, nil],
      9348 => [0, 0, nil, "(17)", nil, nil, nil],
      9349 => [0, 0, nil, "(18)", nil, nil, nil],
      9350 => [0, 0, nil, "(19)", nil, nil, nil],
      9351 => [0, 0, nil, "(20)", nil, nil, nil],
      9352 => [0, 0, nil, "1.", nil, nil, nil],
      9353 => [0, 0, nil, "2.", nil, nil, nil],
      9354 => [0, 0, nil, "3.", nil, nil, nil],
      9355 => [0, 0, nil, "4.", nil, nil, nil],
      9356 => [0, 0, nil, "5.", nil, nil, nil],
      9357 => [0, 0, nil, "6.", nil, nil, nil],
      9358 => [0, 0, nil, "7.", nil, nil, nil],
      9359 => [0, 0, nil, "8.", nil, nil, nil],
      9360 => [0, 0, nil, "9.", nil, nil, nil],
      9361 => [0, 0, nil, "10.", nil, nil, nil],
      9362 => [0, 0, nil, "11.", nil, nil, nil],
      9363 => [0, 0, nil, "12.", nil, nil, nil],
      9364 => [0, 0, nil, "13.", nil, nil, nil],
      9365 => [0, 0, nil, "14.", nil, nil, nil],
      9366 => [0, 0, nil, "15.", nil, nil, nil],
      9367 => [0, 0, nil, "16.", nil, nil, nil],
      9368 => [0, 0, nil, "17.", nil, nil, nil],
      9369 => [0, 0, nil, "18.", nil, nil, nil],
      9370 => [0, 0, nil, "19.", nil, nil, nil],
      9371 => [0, 0, nil, "20.", nil, nil, nil],
      9372 => [0, 0, nil, "(a)", nil, nil, nil],
      9373 => [0, 0, nil, "(b)", nil, nil, nil],
      9374 => [0, 0, nil, "(c)", nil, nil, nil],
      9375 => [0, 0, nil, "(d)", nil, nil, nil],
      9376 => [0, 0, nil, "(e)", nil, nil, nil],
      9377 => [0, 0, nil, "(f)", nil, nil, nil],
      9378 => [0, 0, nil, "(g)", nil, nil, nil],
      9379 => [0, 0, nil, "(h)", nil, nil, nil],
      9380 => [0, 0, nil, "(i)", nil, nil, nil],
      9381 => [0, 0, nil, "(j)", nil, nil, nil],
      9382 => [0, 0, nil, "(k)", nil, nil, nil],
      9383 => [0, 0, nil, "(l)", nil, nil, nil],
      9384 => [0, 0, nil, "(m)", nil, nil, nil],
      9385 => [0, 0, nil, "(n)", nil, nil, nil],
      9386 => [0, 0, nil, "(o)", nil, nil, nil],
      9387 => [0, 0, nil, "(p)", nil, nil, nil],
      9388 => [0, 0, nil, "(q)", nil, nil, nil],
      9389 => [0, 0, nil, "(r)", nil, nil, nil],
      9390 => [0, 0, nil, "(s)", nil, nil, nil],
      9391 => [0, 0, nil, "(t)", nil, nil, nil],
      9392 => [0, 0, nil, "(u)", nil, nil, nil],
      9393 => [0, 0, nil, "(v)", nil, nil, nil],
      9394 => [0, 0, nil, "(w)", nil, nil, nil],
      9395 => [0, 0, nil, "(x)", nil, nil, nil],
      9396 => [0, 0, nil, "(y)", nil, nil, nil],
      9397 => [0, 0, nil, "(z)", nil, nil, nil],
      9398 => [0, 0, nil, "A", nil, 9424, nil],
      9399 => [0, 0, nil, "B", nil, 9425, nil],
      9400 => [0, 0, nil, "C", nil, 9426, nil],
      9401 => [0, 0, nil, "D", nil, 9427, nil],
      9402 => [0, 0, nil, "E", nil, 9428, nil],
      9403 => [0, 0, nil, "F", nil, 9429, nil],
      9404 => [0, 0, nil, "G", nil, 9430, nil],
      9405 => [0, 0, nil, "H", nil, 9431, nil],
      9406 => [0, 0, nil, "I", nil, 9432, nil],
      9407 => [0, 0, nil, "J", nil, 9433, nil],
      9408 => [0, 0, nil, "K", nil, 9434, nil],
      9409 => [0, 0, nil, "L", nil, 9435, nil],
      9410 => [0, 0, nil, "M", nil, 9436, nil],
      9411 => [0, 0, nil, "N", nil, 9437, nil],
      9412 => [0, 0, nil, "O", nil, 9438, nil],
      9413 => [0, 0, nil, "P", nil, 9439, nil],
      9414 => [0, 0, nil, "Q", nil, 9440, nil],
      9415 => [0, 0, nil, "R", nil, 9441, nil],
      9416 => [0, 0, nil, "S", nil, 9442, nil],
      9417 => [0, 0, nil, "T", nil, 9443, nil],
      9418 => [0, 0, nil, "U", nil, 9444, nil],
      9419 => [0, 0, nil, "V", nil, 9445, nil],
      9420 => [0, 0, nil, "W", nil, 9446, nil],
      9421 => [0, 0, nil, "X", nil, 9447, nil],
      9422 => [0, 0, nil, "Y", nil, 9448, nil],
      9423 => [0, 0, nil, "Z", nil, 9449, nil],
      9424 => [0, 0, nil, "a", 9398, nil, 9398],
      9425 => [0, 0, nil, "b", 9399, nil, 9399],
      9426 => [0, 0, nil, "c", 9400, nil, 9400],
      9427 => [0, 0, nil, "d", 9401, nil, 9401],
      9428 => [0, 0, nil, "e", 9402, nil, 9402],
      9429 => [0, 0, nil, "f", 9403, nil, 9403],
      9430 => [0, 0, nil, "g", 9404, nil, 9404],
      9431 => [0, 0, nil, "h", 9405, nil, 9405],
      9432 => [0, 0, nil, "i", 9406, nil, 9406],
      9433 => [0, 0, nil, "j", 9407, nil, 9407],
      9434 => [0, 0, nil, "k", 9408, nil, 9408],
      9435 => [0, 0, nil, "l", 9409, nil, 9409],
      9436 => [0, 0, nil, "m", 9410, nil, 9410],
      9437 => [0, 0, nil, "n", 9411, nil, 9411],
      9438 => [0, 0, nil, "o", 9412, nil, 9412],
      9439 => [0, 0, nil, "p", 9413, nil, 9413],
      9440 => [0, 0, nil, "q", 9414, nil, 9414],
      9441 => [0, 0, nil, "r", 9415, nil, 9415],
      9442 => [0, 0, nil, "s", 9416, nil, 9416],
      9443 => [0, 0, nil, "t", 9417, nil, 9417],
      9444 => [0, 0, nil, "u", 9418, nil, 9418],
      9445 => [0, 0, nil, "v", 9419, nil, 9419],
      9446 => [0, 0, nil, "w", 9420, nil, 9420],
      9447 => [0, 0, nil, "x", 9421, nil, 9421],
      9448 => [0, 0, nil, "y", 9422, nil, 9422],
      9449 => [0, 0, nil, "z", 9423, nil, 9423],
      9450 => [0, 0, nil, "0", nil, nil, nil],
      11935 => [0, 0, nil, "母", nil, nil, nil],
      12019 => [0, 0, nil, "龟", nil, nil, nil],
      12032 => [0, 0, nil, "一", nil, nil, nil],
      12033 => [0, 0, nil, "丨", nil, nil, nil],
      12034 => [0, 0, nil, "丶", nil, nil, nil],
      12035 => [0, 0, nil, "丿", nil, nil, nil],
      12036 => [0, 0, nil, "乙", nil, nil, nil],
      12037 => [0, 0, nil, "亅", nil, nil, nil],
      12038 => [0, 0, nil, "二", nil, nil, nil],
      12039 => [0, 0, nil, "亠", nil, nil, nil],
      12040 => [0, 0, nil, "人", nil, nil, nil],
      12041 => [0, 0, nil, "儿", nil, nil, nil],
      12042 => [0, 0, nil, "入", nil, nil, nil],
      12043 => [0, 0, nil, "八", nil, nil, nil],
      12044 => [0, 0, nil, "冂", nil, nil, nil],
      12045 => [0, 0, nil, "冖", nil, nil, nil],
      12046 => [0, 0, nil, "冫", nil, nil, nil],
      12047 => [0, 0, nil, "几", nil, nil, nil],
      12048 => [0, 0, nil, "凵", nil, nil, nil],
      12049 => [0, 0, nil, "刀", nil, nil, nil],
      12050 => [0, 0, nil, "力", nil, nil, nil],
      12051 => [0, 0, nil, "勹", nil, nil, nil],
      12052 => [0, 0, nil, "匕", nil, nil, nil],
      12053 => [0, 0, nil, "匚", nil, nil, nil],
      12054 => [0, 0, nil, "匸", nil, nil, nil],
      12055 => [0, 0, nil, "十", nil, nil, nil],
      12056 => [0, 0, nil, "卜", nil, nil, nil],
      12057 => [0, 0, nil, "卩", nil, nil, nil],
      12058 => [0, 0, nil, "厂", nil, nil, nil],
      12059 => [0, 0, nil, "厶", nil, nil, nil],
      12060 => [0, 0, nil, "又", nil, nil, nil],
      12061 => [0, 0, nil, "口", nil, nil, nil],
      12062 => [0, 0, nil, "囗", nil, nil, nil],
      12063 => [0, 0, nil, "土", nil, nil, nil],
      12064 => [0, 0, nil, "士", nil, nil, nil],
      12065 => [0, 0, nil, "夂", nil, nil, nil],
      12066 => [0, 0, nil, "夊", nil, nil, nil],
      12067 => [0, 0, nil, "夕", nil, nil, nil],
      12068 => [0, 0, nil, "大", nil, nil, nil],
      12069 => [0, 0, nil, "女", nil, nil, nil],
      12070 => [0, 0, nil, "子", nil, nil, nil],
      12071 => [0, 0, nil, "宀", nil, nil, nil],
      12072 => [0, 0, nil, "寸", nil, nil, nil],
      12073 => [0, 0, nil, "小", nil, nil, nil],
      12074 => [0, 0, nil, "尢", nil, nil, nil],
      12075 => [0, 0, nil, "尸", nil, nil, nil],
      12076 => [0, 0, nil, "屮", nil, nil, nil],
      12077 => [0, 0, nil, "山", nil, nil, nil],
      12078 => [0, 0, nil, "巛", nil, nil, nil],
      12079 => [0, 0, nil, "工", nil, nil, nil],
      12080 => [0, 0, nil, "己", nil, nil, nil],
      12081 => [0, 0, nil, "巾", nil, nil, nil],
      12082 => [0, 0, nil, "干", nil, nil, nil],
      12083 => [0, 0, nil, "幺", nil, nil, nil],
      12084 => [0, 0, nil, "广", nil, nil, nil],
      12085 => [0, 0, nil, "廴", nil, nil, nil],
      12086 => [0, 0, nil, "廾", nil, nil, nil],
      12087 => [0, 0, nil, "弋", nil, nil, nil],
      12088 => [0, 0, nil, "弓", nil, nil, nil],
      12089 => [0, 0, nil, "彐", nil, nil, nil],
      12090 => [0, 0, nil, "彡", nil, nil, nil],
      12091 => [0, 0, nil, "彳", nil, nil, nil],
      12092 => [0, 0, nil, "心", nil, nil, nil],
      12093 => [0, 0, nil, "戈", nil, nil, nil],
      12094 => [0, 0, nil, "戶", nil, nil, nil],
      12095 => [0, 0, nil, "手", nil, nil, nil],
      12096 => [0, 0, nil, "支", nil, nil, nil],
      12097 => [0, 0, nil, "攴", nil, nil, nil],
      12098 => [0, 0, nil, "文", nil, nil, nil],
      12099 => [0, 0, nil, "斗", nil, nil, nil],
      12100 => [0, 0, nil, "斤", nil, nil, nil],
      12101 => [0, 0, nil, "方", nil, nil, nil],
      12102 => [0, 0, nil, "无", nil, nil, nil],
      12103 => [0, 0, nil, "日", nil, nil, nil],
      12104 => [0, 0, nil, "曰", nil, nil, nil],
      12105 => [0, 0, nil, "月", nil, nil, nil],
      12106 => [0, 0, nil, "木", nil, nil, nil],
      12107 => [0, 0, nil, "欠", nil, nil, nil],
      12108 => [0, 0, nil, "止", nil, nil, nil],
      12109 => [0, 0, nil, "歹", nil, nil, nil],
      12110 => [0, 0, nil, "殳", nil, nil, nil],
      12111 => [0, 0, nil, "毋", nil, nil, nil],
      12112 => [0, 0, nil, "比", nil, nil, nil],
      12113 => [0, 0, nil, "毛", nil, nil, nil],
      12114 => [0, 0, nil, "氏", nil, nil, nil],
      12115 => [0, 0, nil, "气", nil, nil, nil],
      12116 => [0, 0, nil, "水", nil, nil, nil],
      12117 => [0, 0, nil, "火", nil, nil, nil],
      12118 => [0, 0, nil, "爪", nil, nil, nil],
      12119 => [0, 0, nil, "父", nil, nil, nil],
      12120 => [0, 0, nil, "爻", nil, nil, nil],
      12121 => [0, 0, nil, "爿", nil, nil, nil],
      12122 => [0, 0, nil, "片", nil, nil, nil],
      12123 => [0, 0, nil, "牙", nil, nil, nil],
      12124 => [0, 0, nil, "牛", nil, nil, nil],
      12125 => [0, 0, nil, "犬", nil, nil, nil],
      12126 => [0, 0, nil, "玄", nil, nil, nil],
      12127 => [0, 0, nil, "玉", nil, nil, nil],
      12128 => [0, 0, nil, "瓜", nil, nil, nil],
      12129 => [0, 0, nil, "瓦", nil, nil, nil],
      12130 => [0, 0, nil, "甘", nil, nil, nil],
      12131 => [0, 0, nil, "生", nil, nil, nil],
      12132 => [0, 0, nil, "用", nil, nil, nil],
      12133 => [0, 0, nil, "田", nil, nil, nil],
      12134 => [0, 0, nil, "疋", nil, nil, nil],
      12135 => [0, 0, nil, "疒", nil, nil, nil],
      12136 => [0, 0, nil, "癶", nil, nil, nil],
      12137 => [0, 0, nil, "白", nil, nil, nil],
      12138 => [0, 0, nil, "皮", nil, nil, nil],
      12139 => [0, 0, nil, "皿", nil, nil, nil],
      12140 => [0, 0, nil, "目", nil, nil, nil],
      12141 => [0, 0, nil, "矛", nil, nil, nil],
      12142 => [0, 0, nil, "矢", nil, nil, nil],
      12143 => [0, 0, nil, "石", nil, nil, nil],
      12144 => [0, 0, nil, "示", nil, nil, nil],
      12145 => [0, 0, nil, "禸", nil, nil, nil],
      12146 => [0, 0, nil, "禾", nil, nil, nil],
      12147 => [0, 0, nil, "穴", nil, nil, nil],
      12148 => [0, 0, nil, "立", nil, nil, nil],
      12149 => [0, 0, nil, "竹", nil, nil, nil],
      12150 => [0, 0, nil, "米", nil, nil, nil],
      12151 => [0, 0, nil, "糸", nil, nil, nil],
      12152 => [0, 0, nil, "缶", nil, nil, nil],
      12153 => [0, 0, nil, "网", nil, nil, nil],
      12154 => [0, 0, nil, "羊", nil, nil, nil],
      12155 => [0, 0, nil, "羽", nil, nil, nil],
      12156 => [0, 0, nil, "老", nil, nil, nil],
      12157 => [0, 0, nil, "而", nil, nil, nil],
      12158 => [0, 0, nil, "耒", nil, nil, nil],
      12159 => [0, 0, nil, "耳", nil, nil, nil],
      12160 => [0, 0, nil, "聿", nil, nil, nil],
      12161 => [0, 0, nil, "肉", nil, nil, nil],
      12162 => [0, 0, nil, "臣", nil, nil, nil],
      12163 => [0, 0, nil, "自", nil, nil, nil],
      12164 => [0, 0, nil, "至", nil, nil, nil],
      12165 => [0, 0, nil, "臼", nil, nil, nil],
      12166 => [0, 0, nil, "舌", nil, nil, nil],
      12167 => [0, 0, nil, "舛", nil, nil, nil],
      12168 => [0, 0, nil, "舟", nil, nil, nil],
      12169 => [0, 0, nil, "艮", nil, nil, nil],
      12170 => [0, 0, nil, "色", nil, nil, nil],
      12171 => [0, 0, nil, "艸", nil, nil, nil],
      12172 => [0, 0, nil, "虍", nil, nil, nil],
      12173 => [0, 0, nil, "虫", nil, nil, nil],
      12174 => [0, 0, nil, "血", nil, nil, nil],
      12175 => [0, 0, nil, "行", nil, nil, nil],
      12176 => [0, 0, nil, "衣", nil, nil, nil],
      12177 => [0, 0, nil, "襾", nil, nil, nil],
      12178 => [0, 0, nil, "見", nil, nil, nil],
      12179 => [0, 0, nil, "角", nil, nil, nil],
      12180 => [0, 0, nil, "言", nil, nil, nil],
      12181 => [0, 0, nil, "谷", nil, nil, nil],
      12182 => [0, 0, nil, "豆", nil, nil, nil],
      12183 => [0, 0, nil, "豕", nil, nil, nil],
      12184 => [0, 0, nil, "豸", nil, nil, nil],
      12185 => [0, 0, nil, "貝", nil, nil, nil],
      12186 => [0, 0, nil, "赤", nil, nil, nil],
      12187 => [0, 0, nil, "走", nil, nil, nil],
      12188 => [0, 0, nil, "足", nil, nil, nil],
      12189 => [0, 0, nil, "身", nil, nil, nil],
      12190 => [0, 0, nil, "車", nil, nil, nil],
      12191 => [0, 0, nil, "辛", nil, nil, nil],
      12192 => [0, 0, nil, "辰", nil, nil, nil],
      12193 => [0, 0, nil, "辵", nil, nil, nil],
      12194 => [0, 0, nil, "邑", nil, nil, nil],
      12195 => [0, 0, nil, "酉", nil, nil, nil],
      12196 => [0, 0, nil, "釆", nil, nil, nil],
      12197 => [0, 0, nil, "里", nil, nil, nil],
      12198 => [0, 0, nil, "金", nil, nil, nil],
      12199 => [0, 0, nil, "長", nil, nil, nil],
      12200 => [0, 0, nil, "門", nil, nil, nil],
      12201 => [0, 0, nil, "阜", nil, nil, nil],
      12202 => [0, 0, nil, "隶", nil, nil, nil],
      12203 => [0, 0, nil, "隹", nil, nil, nil],
      12204 => [0, 0, nil, "雨", nil, nil, nil],
      12205 => [0, 0, nil, "靑", nil, nil, nil],
      12206 => [0, 0, nil, "非", nil, nil, nil],
      12207 => [0, 0, nil, "面", nil, nil, nil],
      12208 => [0, 0, nil, "革", nil, nil, nil],
      12209 => [0, 0, nil, "韋", nil, nil, nil],
      12210 => [0, 0, nil, "韭", nil, nil, nil],
      12211 => [0, 0, nil, "音", nil, nil, nil],
      12212 => [0, 0, nil, "頁", nil, nil, nil],
      12213 => [0, 0, nil, "風", nil, nil, nil],
      12214 => [0, 0, nil, "飛", nil, nil, nil],
      12215 => [0, 0, nil, "食", nil, nil, nil],
      12216 => [0, 0, nil, "首", nil, nil, nil],
      12217 => [0, 0, nil, "香", nil, nil, nil],
      12218 => [0, 0, nil, "馬", nil, nil, nil],
      12219 => [0, 0, nil, "骨", nil, nil, nil],
      12220 => [0, 0, nil, "高", nil, nil, nil],
      12221 => [0, 0, nil, "髟", nil, nil, nil],
      12222 => [0, 0, nil, "鬥", nil, nil, nil],
      12223 => [0, 0, nil, "鬯", nil, nil, nil],
      12224 => [0, 0, nil, "鬲", nil, nil, nil],
      12225 => [0, 0, nil, "鬼", nil, nil, nil],
      12226 => [0, 0, nil, "魚", nil, nil, nil],
      12227 => [0, 0, nil, "鳥", nil, nil, nil],
      12228 => [0, 0, nil, "鹵", nil, nil, nil],
      12229 => [0, 0, nil, "鹿", nil, nil, nil],
      12230 => [0, 0, nil, "麥", nil, nil, nil],
      12231 => [0, 0, nil, "麻", nil, nil, nil],
      12232 => [0, 0, nil, "黃", nil, nil, nil],
      12233 => [0, 0, nil, "黍", nil, nil, nil],
      12234 => [0, 0, nil, "黑", nil, nil, nil],
      12235 => [0, 0, nil, "黹", nil, nil, nil],
      12236 => [0, 0, nil, "黽", nil, nil, nil],
      12237 => [0, 0, nil, "鼎", nil, nil, nil],
      12238 => [0, 0, nil, "鼓", nil, nil, nil],
      12239 => [0, 0, nil, "鼠", nil, nil, nil],
      12240 => [0, 0, nil, "鼻", nil, nil, nil],
      12241 => [0, 0, nil, "齊", nil, nil, nil],
      12242 => [0, 0, nil, "齒", nil, nil, nil],
      12243 => [0, 0, nil, "龍", nil, nil, nil],
      12244 => [0, 0, nil, "龜", nil, nil, nil],
      12245 => [0, 0, nil, "龠", nil, nil, nil],
      12288 => [0, 0, nil, " ", nil, nil, nil],
      12330 => [218, 0, nil, nil, nil, nil, nil],
      12331 => [228, 0, nil, nil, nil, nil, nil],
      12332 => [232, 0, nil, nil, nil, nil, nil],
      12333 => [222, 0, nil, nil, nil, nil, nil],
      12334 => [224, 0, nil, nil, nil, nil, nil],
      12335 => [224, 0, nil, nil, nil, nil, nil],
      12342 => [0, 0, nil, "〒", nil, nil, nil],
      12344 => [0, 0, nil, "十", nil, nil, nil],
      12345 => [0, 0, nil, "卄", nil, nil, nil],
      12346 => [0, 0, nil, "卅", nil, nil, nil],
      12364 => [0, 0, "が", "が", nil, nil, nil],
      12366 => [0, 0, "ぎ", "ぎ", nil, nil, nil],
      12368 => [0, 0, "ぐ", "ぐ", nil, nil, nil],
      12370 => [0, 0, "げ", "げ", nil, nil, nil],
      12372 => [0, 0, "ご", "ご", nil, nil, nil],
      12374 => [0, 0, "ざ", "ざ", nil, nil, nil],
      12376 => [0, 0, "じ", "じ", nil, nil, nil],
      12378 => [0, 0, "ず", "ず", nil, nil, nil],
      12380 => [0, 0, "ぜ", "ぜ", nil, nil, nil],
      12382 => [0, 0, "ぞ", "ぞ", nil, nil, nil],
      12384 => [0, 0, "だ", "だ", nil, nil, nil],
      12386 => [0, 0, "ぢ", "ぢ", nil, nil, nil],
      12389 => [0, 0, "づ", "づ", nil, nil, nil],
      12391 => [0, 0, "で", "で", nil, nil, nil],
      12393 => [0, 0, "ど", "ど", nil, nil, nil],
      12400 => [0, 0, "ば", "ば", nil, nil, nil],
      12401 => [0, 0, "ぱ", "ぱ", nil, nil, nil],
      12403 => [0, 0, "び", "び", nil, nil, nil],
      12404 => [0, 0, "ぴ", "ぴ", nil, nil, nil],
      12406 => [0, 0, "ぶ", "ぶ", nil, nil, nil],
      12407 => [0, 0, "ぷ", "ぷ", nil, nil, nil],
      12409 => [0, 0, "べ", "べ", nil, nil, nil],
      12410 => [0, 0, "ぺ", "ぺ", nil, nil, nil],
      12412 => [0, 0, "ぼ", "ぼ", nil, nil, nil],
      12413 => [0, 0, "ぽ", "ぽ", nil, nil, nil],
      12436 => [0, 0, "ゔ", "ゔ", nil, nil, nil],
      12441 => [8, 0, nil, nil, nil, nil, nil],
      12442 => [8, 0, nil, nil, nil, nil, nil],
      12443 => [0, 0, nil, " ゙", nil, nil, nil],
      12444 => [0, 0, nil, " ゚", nil, nil, nil],
      12446 => [0, 0, "ゞ", "ゞ", nil, nil, nil],
      12460 => [0, 0, "ガ", "ガ", nil, nil, nil],
      12462 => [0, 0, "ギ", "ギ", nil, nil, nil],
      12464 => [0, 0, "グ", "グ", nil, nil, nil],
      12466 => [0, 0, "ゲ", "ゲ", nil, nil, nil],
      12468 => [0, 0, "ゴ", "ゴ", nil, nil, nil],
      12470 => [0, 0, "ザ", "ザ", nil, nil, nil],
      12472 => [0, 0, "ジ", "ジ", nil, nil, nil],
      12474 => [0, 0, "ズ", "ズ", nil, nil, nil],
      12476 => [0, 0, "ゼ", "ゼ", nil, nil, nil],
      12478 => [0, 0, "ゾ", "ゾ", nil, nil, nil],
      12480 => [0, 0, "ダ", "ダ", nil, nil, nil],
      12482 => [0, 0, "ヂ", "ヂ", nil, nil, nil],
      12485 => [0, 0, "ヅ", "ヅ", nil, nil, nil],
      12487 => [0, 0, "デ", "デ", nil, nil, nil],
      12489 => [0, 0, "ド", "ド", nil, nil, nil],
      12496 => [0, 0, "バ", "バ", nil, nil, nil],
      12497 => [0, 0, "パ", "パ", nil, nil, nil],
      12499 => [0, 0, "ビ", "ビ", nil, nil, nil],
      12500 => [0, 0, "ピ", "ピ", nil, nil, nil],
      12502 => [0, 0, "ブ", "ブ", nil, nil, nil],
      12503 => [0, 0, "プ", "プ", nil, nil, nil],
      12505 => [0, 0, "ベ", "ベ", nil, nil, nil],
      12506 => [0, 0, "ペ", "ペ", nil, nil, nil],
      12508 => [0, 0, "ボ", "ボ", nil, nil, nil],
      12509 => [0, 0, "ポ", "ポ", nil, nil, nil],
      12532 => [0, 0, "ヴ", "ヴ", nil, nil, nil],
      12535 => [0, 0, "ヷ", "ヷ", nil, nil, nil],
      12536 => [0, 0, "ヸ", "ヸ", nil, nil, nil],
      12537 => [0, 0, "ヹ", "ヹ", nil, nil, nil],
      12538 => [0, 0, "ヺ", "ヺ", nil, nil, nil],
      12542 => [0, 0, "ヾ", "ヾ", nil, nil, nil],
      12593 => [0, 0, nil, "ᄀ", nil, nil, nil],
      12594 => [0, 0, nil, "ᄁ", nil, nil, nil],
      12595 => [0, 0, nil, "ᆪ", nil, nil, nil],
      12596 => [0, 0, nil, "ᄂ", nil, nil, nil],
      12597 => [0, 0, nil, "ᆬ", nil, nil, nil],
      12598 => [0, 0, nil, "ᆭ", nil, nil, nil],
      12599 => [0, 0, nil, "ᄃ", nil, nil, nil],
      12600 => [0, 0, nil, "ᄄ", nil, nil, nil],
      12601 => [0, 0, nil, "ᄅ", nil, nil, nil],
      12602 => [0, 0, nil, "ᆰ", nil, nil, nil],
      12603 => [0, 0, nil, "ᆱ", nil, nil, nil],
      12604 => [0, 0, nil, "ᆲ", nil, nil, nil],
      12605 => [0, 0, nil, "ᆳ", nil, nil, nil],
      12606 => [0, 0, nil, "ᆴ", nil, nil, nil],
      12607 => [0, 0, nil, "ᆵ", nil, nil, nil],
      12608 => [0, 0, nil, "ᄚ", nil, nil, nil],
      12609 => [0, 0, nil, "ᄆ", nil, nil, nil],
      12610 => [0, 0, nil, "ᄇ", nil, nil, nil],
      12611 => [0, 0, nil, "ᄈ", nil, nil, nil],
      12612 => [0, 0, nil, "ᄡ", nil, nil, nil],
      12613 => [0, 0, nil, "ᄉ", nil, nil, nil],
      12614 => [0, 0, nil, "ᄊ", nil, nil, nil],
      12615 => [0, 0, nil, "ᄋ", nil, nil, nil],
      12616 => [0, 0, nil, "ᄌ", nil, nil, nil],
      12617 => [0, 0, nil, "ᄍ", nil, nil, nil],
      12618 => [0, 0, nil, "ᄎ", nil, nil, nil],
      12619 => [0, 0, nil, "ᄏ", nil, nil, nil],
      12620 => [0, 0, nil, "ᄐ", nil, nil, nil],
      12621 => [0, 0, nil, "ᄑ", nil, nil, nil],
      12622 => [0, 0, nil, "ᄒ", nil, nil, nil],
      12623 => [0, 0, nil, "ᅡ", nil, nil, nil],
      12624 => [0, 0, nil, "ᅢ", nil, nil, nil],
      12625 => [0, 0, nil, "ᅣ", nil, nil, nil],
      12626 => [0, 0, nil, "ᅤ", nil, nil, nil],
      12627 => [0, 0, nil, "ᅥ", nil, nil, nil],
      12628 => [0, 0, nil, "ᅦ", nil, nil, nil],
      12629 => [0, 0, nil, "ᅧ", nil, nil, nil],
      12630 => [0, 0, nil, "ᅨ", nil, nil, nil],
      12631 => [0, 0, nil, "ᅩ", nil, nil, nil],
      12632 => [0, 0, nil, "ᅪ", nil, nil, nil],
      12633 => [0, 0, nil, "ᅫ", nil, nil, nil],
      12634 => [0, 0, nil, "ᅬ", nil, nil, nil],
      12635 => [0, 0, nil, "ᅭ", nil, nil, nil],
      12636 => [0, 0, nil, "ᅮ", nil, nil, nil],
      12637 => [0, 0, nil, "ᅯ", nil, nil, nil],
      12638 => [0, 0, nil, "ᅰ", nil, nil, nil],
      12639 => [0, 0, nil, "ᅱ", nil, nil, nil],
      12640 => [0, 0, nil, "ᅲ", nil, nil, nil],
      12641 => [0, 0, nil, "ᅳ", nil, nil, nil],
      12642 => [0, 0, nil, "ᅴ", nil, nil, nil],
      12643 => [0, 0, nil, "ᅵ", nil, nil, nil],
      12644 => [0, 0, nil, "ᅠ", nil, nil, nil],
      12645 => [0, 0, nil, "ᄔ", nil, nil, nil],
      12646 => [0, 0, nil, "ᄕ", nil, nil, nil],
      12647 => [0, 0, nil, "ᇇ", nil, nil, nil],
      12648 => [0, 0, nil, "ᇈ", nil, nil, nil],
      12649 => [0, 0, nil, "ᇌ", nil, nil, nil],
      12650 => [0, 0, nil, "ᇎ", nil, nil, nil],
      12651 => [0, 0, nil, "ᇓ", nil, nil, nil],
      12652 => [0, 0, nil, "ᇗ", nil, nil, nil],
      12653 => [0, 0, nil, "ᇙ", nil, nil, nil],
      12654 => [0, 0, nil, "ᄜ", nil, nil, nil],
      12655 => [0, 0, nil, "ᇝ", nil, nil, nil],
      12656 => [0, 0, nil, "ᇟ", nil, nil, nil],
      12657 => [0, 0, nil, "ᄝ", nil, nil, nil],
      12658 => [0, 0, nil, "ᄞ", nil, nil, nil],
      12659 => [0, 0, nil, "ᄠ", nil, nil, nil],
      12660 => [0, 0, nil, "ᄢ", nil, nil, nil],
      12661 => [0, 0, nil, "ᄣ", nil, nil, nil],
      12662 => [0, 0, nil, "ᄧ", nil, nil, nil],
      12663 => [0, 0, nil, "ᄩ", nil, nil, nil],
      12664 => [0, 0, nil, "ᄫ", nil, nil, nil],
      12665 => [0, 0, nil, "ᄬ", nil, nil, nil],
      12666 => [0, 0, nil, "ᄭ", nil, nil, nil],
      12667 => [0, 0, nil, "ᄮ", nil, nil, nil],
      12668 => [0, 0, nil, "ᄯ", nil, nil, nil],
      12669 => [0, 0, nil, "ᄲ", nil, nil, nil],
      12670 => [0, 0, nil, "ᄶ", nil, nil, nil],
      12671 => [0, 0, nil, "ᅀ", nil, nil, nil],
      12672 => [0, 0, nil, "ᅇ", nil, nil, nil],
      12673 => [0, 0, nil, "ᅌ", nil, nil, nil],
      12674 => [0, 0, nil, "ᇱ", nil, nil, nil],
      12675 => [0, 0, nil, "ᇲ", nil, nil, nil],
      12676 => [0, 0, nil, "ᅗ", nil, nil, nil],
      12677 => [0, 0, nil, "ᅘ", nil, nil, nil],
      12678 => [0, 0, nil, "ᅙ", nil, nil, nil],
      12679 => [0, 0, nil, "ᆄ", nil, nil, nil],
      12680 => [0, 0, nil, "ᆅ", nil, nil, nil],
      12681 => [0, 0, nil, "ᆈ", nil, nil, nil],
      12682 => [0, 0, nil, "ᆑ", nil, nil, nil],
      12683 => [0, 0, nil, "ᆒ", nil, nil, nil],
      12684 => [0, 0, nil, "ᆔ", nil, nil, nil],
      12685 => [0, 0, nil, "ᆞ", nil, nil, nil],
      12686 => [0, 0, nil, "ᆡ", nil, nil, nil],
      12690 => [0, 0, nil, "一", nil, nil, nil],
      12691 => [0, 0, nil, "二", nil, nil, nil],
      12692 => [0, 0, nil, "三", nil, nil, nil],
      12693 => [0, 0, nil, "四", nil, nil, nil],
      12694 => [0, 0, nil, "上", nil, nil, nil],
      12695 => [0, 0, nil, "中", nil, nil, nil],
      12696 => [0, 0, nil, "下", nil, nil, nil],
      12697 => [0, 0, nil, "甲", nil, nil, nil],
      12698 => [0, 0, nil, "乙", nil, nil, nil],
      12699 => [0, 0, nil, "丙", nil, nil, nil],
      12700 => [0, 0, nil, "丁", nil, nil, nil],
      12701 => [0, 0, nil, "天", nil, nil, nil],
      12702 => [0, 0, nil, "地", nil, nil, nil],
      12703 => [0, 0, nil, "人", nil, nil, nil],
      12800 => [0, 0, nil, "(ᄀ)", nil, nil, nil],
      12801 => [0, 0, nil, "(ᄂ)", nil, nil, nil],
      12802 => [0, 0, nil, "(ᄃ)", nil, nil, nil],
      12803 => [0, 0, nil, "(ᄅ)", nil, nil, nil],
      12804 => [0, 0, nil, "(ᄆ)", nil, nil, nil],
      12805 => [0, 0, nil, "(ᄇ)", nil, nil, nil],
      12806 => [0, 0, nil, "(ᄉ)", nil, nil, nil],
      12807 => [0, 0, nil, "(ᄋ)", nil, nil, nil],
      12808 => [0, 0, nil, "(ᄌ)", nil, nil, nil],
      12809 => [0, 0, nil, "(ᄎ)", nil, nil, nil],
      12810 => [0, 0, nil, "(ᄏ)", nil, nil, nil],
      12811 => [0, 0, nil, "(ᄐ)", nil, nil, nil],
      12812 => [0, 0, nil, "(ᄑ)", nil, nil, nil],
      12813 => [0, 0, nil, "(ᄒ)", nil, nil, nil],
      12814 => [0, 0, nil, "(가)", nil, nil, nil],
      12815 => [0, 0, nil, "(나)", nil, nil, nil],
      12816 => [0, 0, nil, "(다)", nil, nil, nil],
      12817 => [0, 0, nil, "(라)", nil, nil, nil],
      12818 => [0, 0, nil, "(마)", nil, nil, nil],
      12819 => [0, 0, nil, "(바)", nil, nil, nil],
      12820 => [0, 0, nil, "(사)", nil, nil, nil],
      12821 => [0, 0, nil, "(아)", nil, nil, nil],
      12822 => [0, 0, nil, "(자)", nil, nil, nil],
      12823 => [0, 0, nil, "(차)", nil, nil, nil],
      12824 => [0, 0, nil, "(카)", nil, nil, nil],
      12825 => [0, 0, nil, "(타)", nil, nil, nil],
      12826 => [0, 0, nil, "(파)", nil, nil, nil],
      12827 => [0, 0, nil, "(하)", nil, nil, nil],
      12828 => [0, 0, nil, "(주)", nil, nil, nil],
      12832 => [0, 0, nil, "(一)", nil, nil, nil],
      12833 => [0, 0, nil, "(二)", nil, nil, nil],
      12834 => [0, 0, nil, "(三)", nil, nil, nil],
      12835 => [0, 0, nil, "(四)", nil, nil, nil],
      12836 => [0, 0, nil, "(五)", nil, nil, nil],
      12837 => [0, 0, nil, "(六)", nil, nil, nil],
      12838 => [0, 0, nil, "(七)", nil, nil, nil],
      12839 => [0, 0, nil, "(八)", nil, nil, nil],
      12840 => [0, 0, nil, "(九)", nil, nil, nil],
      12841 => [0, 0, nil, "(十)", nil, nil, nil],
      12842 => [0, 0, nil, "(月)", nil, nil, nil],
      12843 => [0, 0, nil, "(火)", nil, nil, nil],
      12844 => [0, 0, nil, "(水)", nil, nil, nil],
      12845 => [0, 0, nil, "(木)", nil, nil, nil],
      12846 => [0, 0, nil, "(金)", nil, nil, nil],
      12847 => [0, 0, nil, "(土)", nil, nil, nil],
      12848 => [0, 0, nil, "(日)", nil, nil, nil],
      12849 => [0, 0, nil, "(株)", nil, nil, nil],
      12850 => [0, 0, nil, "(有)", nil, nil, nil],
      12851 => [0, 0, nil, "(社)", nil, nil, nil],
      12852 => [0, 0, nil, "(名)", nil, nil, nil],
      12853 => [0, 0, nil, "(特)", nil, nil, nil],
      12854 => [0, 0, nil, "(財)", nil, nil, nil],
      12855 => [0, 0, nil, "(祝)", nil, nil, nil],
      12856 => [0, 0, nil, "(労)", nil, nil, nil],
      12857 => [0, 0, nil, "(代)", nil, nil, nil],
      12858 => [0, 0, nil, "(呼)", nil, nil, nil],
      12859 => [0, 0, nil, "(学)", nil, nil, nil],
      12860 => [0, 0, nil, "(監)", nil, nil, nil],
      12861 => [0, 0, nil, "(企)", nil, nil, nil],
      12862 => [0, 0, nil, "(資)", nil, nil, nil],
      12863 => [0, 0, nil, "(協)", nil, nil, nil],
      12864 => [0, 0, nil, "(祭)", nil, nil, nil],
      12865 => [0, 0, nil, "(休)", nil, nil, nil],
      12866 => [0, 0, nil, "(自)", nil, nil, nil],
      12867 => [0, 0, nil, "(至)", nil, nil, nil],
      12896 => [0, 0, nil, "ᄀ", nil, nil, nil],
      12897 => [0, 0, nil, "ᄂ", nil, nil, nil],
      12898 => [0, 0, nil, "ᄃ", nil, nil, nil],
      12899 => [0, 0, nil, "ᄅ", nil, nil, nil],
      12900 => [0, 0, nil, "ᄆ", nil, nil, nil],
      12901 => [0, 0, nil, "ᄇ", nil, nil, nil],
      12902 => [0, 0, nil, "ᄉ", nil, nil, nil],
      12903 => [0, 0, nil, "ᄋ", nil, nil, nil],
      12904 => [0, 0, nil, "ᄌ", nil, nil, nil],
      12905 => [0, 0, nil, "ᄎ", nil, nil, nil],
      12906 => [0, 0, nil, "ᄏ", nil, nil, nil],
      12907 => [0, 0, nil, "ᄐ", nil, nil, nil],
      12908 => [0, 0, nil, "ᄑ", nil, nil, nil],
      12909 => [0, 0, nil, "ᄒ", nil, nil, nil],
      12910 => [0, 0, nil, "가", nil, nil, nil],
      12911 => [0, 0, nil, "나", nil, nil, nil],
      12912 => [0, 0, nil, "다", nil, nil, nil],
      12913 => [0, 0, nil, "라", nil, nil, nil],
      12914 => [0, 0, nil, "마", nil, nil, nil],
      12915 => [0, 0, nil, "바", nil, nil, nil],
      12916 => [0, 0, nil, "사", nil, nil, nil],
      12917 => [0, 0, nil, "아", nil, nil, nil],
      12918 => [0, 0, nil, "자", nil, nil, nil],
      12919 => [0, 0, nil, "차", nil, nil, nil],
      12920 => [0, 0, nil, "카", nil, nil, nil],
      12921 => [0, 0, nil, "타", nil, nil, nil],
      12922 => [0, 0, nil, "파", nil, nil, nil],
      12923 => [0, 0, nil, "하", nil, nil, nil],
      12928 => [0, 0, nil, "一", nil, nil, nil],
      12929 => [0, 0, nil, "二", nil, nil, nil],
      12930 => [0, 0, nil, "三", nil, nil, nil],
      12931 => [0, 0, nil, "四", nil, nil, nil],
      12932 => [0, 0, nil, "五", nil, nil, nil],
      12933 => [0, 0, nil, "六", nil, nil, nil],
      12934 => [0, 0, nil, "七", nil, nil, nil],
      12935 => [0, 0, nil, "八", nil, nil, nil],
      12936 => [0, 0, nil, "九", nil, nil, nil],
      12937 => [0, 0, nil, "十", nil, nil, nil],
      12938 => [0, 0, nil, "月", nil, nil, nil],
      12939 => [0, 0, nil, "火", nil, nil, nil],
      12940 => [0, 0, nil, "水", nil, nil, nil],
      12941 => [0, 0, nil, "木", nil, nil, nil],
      12942 => [0, 0, nil, "金", nil, nil, nil],
      12943 => [0, 0, nil, "土", nil, nil, nil],
      12944 => [0, 0, nil, "日", nil, nil, nil],
      12945 => [0, 0, nil, "株", nil, nil, nil],
      12946 => [0, 0, nil, "有", nil, nil, nil],
      12947 => [0, 0, nil, "社", nil, nil, nil],
      12948 => [0, 0, nil, "名", nil, nil, nil],
      12949 => [0, 0, nil, "特", nil, nil, nil],
      12950 => [0, 0, nil, "財", nil, nil, nil],
      12951 => [0, 0, nil, "祝", nil, nil, nil],
      12952 => [0, 0, nil, "労", nil, nil, nil],
      12953 => [0, 0, nil, "秘", nil, nil, nil],
      12954 => [0, 0, nil, "男", nil, nil, nil],
      12955 => [0, 0, nil, "女", nil, nil, nil],
      12956 => [0, 0, nil, "適", nil, nil, nil],
      12957 => [0, 0, nil, "優", nil, nil, nil],
      12958 => [0, 0, nil, "印", nil, nil, nil],
      12959 => [0, 0, nil, "注", nil, nil, nil],
      12960 => [0, 0, nil, "項", nil, nil, nil],
      12961 => [0, 0, nil, "休", nil, nil, nil],
      12962 => [0, 0, nil, "写", nil, nil, nil],
      12963 => [0, 0, nil, "正", nil, nil, nil],
      12964 => [0, 0, nil, "上", nil, nil, nil],
      12965 => [0, 0, nil, "中", nil, nil, nil],
      12966 => [0, 0, nil, "下", nil, nil, nil],
      12967 => [0, 0, nil, "左", nil, nil, nil],
      12968 => [0, 0, nil, "右", nil, nil, nil],
      12969 => [0, 0, nil, "医", nil, nil, nil],
      12970 => [0, 0, nil, "宗", nil, nil, nil],
      12971 => [0, 0, nil, "学", nil, nil, nil],
      12972 => [0, 0, nil, "監", nil, nil, nil],
      12973 => [0, 0, nil, "企", nil, nil, nil],
      12974 => [0, 0, nil, "資", nil, nil, nil],
      12975 => [0, 0, nil, "協", nil, nil, nil],
      12976 => [0, 0, nil, "夜", nil, nil, nil],
      12992 => [0, 0, nil, "1月", nil, nil, nil],
      12993 => [0, 0, nil, "2月", nil, nil, nil],
      12994 => [0, 0, nil, "3月", nil, nil, nil],
      12995 => [0, 0, nil, "4月", nil, nil, nil],
      12996 => [0, 0, nil, "5月", nil, nil, nil],
      12997 => [0, 0, nil, "6月", nil, nil, nil],
      12998 => [0, 0, nil, "7月", nil, nil, nil],
      12999 => [0, 0, nil, "8月", nil, nil, nil],
      13000 => [0, 0, nil, "9月", nil, nil, nil],
      13001 => [0, 0, nil, "10月", nil, nil, nil],
      13002 => [0, 0, nil, "11月", nil, nil, nil],
      13003 => [0, 0, nil, "12月", nil, nil, nil],
      13008 => [0, 0, nil, "ア", nil, nil, nil],
      13009 => [0, 0, nil, "イ", nil, nil, nil],
      13010 => [0, 0, nil, "ウ", nil, nil, nil],
      13011 => [0, 0, nil, "エ", nil, nil, nil],
      13012 => [0, 0, nil, "オ", nil, nil, nil],
      13013 => [0, 0, nil, "カ", nil, nil, nil],
      13014 => [0, 0, nil, "キ", nil, nil, nil],
      13015 => [0, 0, nil, "ク", nil, nil, nil],
      13016 => [0, 0, nil, "ケ", nil, nil, nil],
      13017 => [0, 0, nil, "コ", nil, nil, nil],
      13018 => [0, 0, nil, "サ", nil, nil, nil],
      13019 => [0, 0, nil, "シ", nil, nil, nil],
      13020 => [0, 0, nil, "ス", nil, nil, nil],
      13021 => [0, 0, nil, "セ", nil, nil, nil],
      13022 => [0, 0, nil, "ソ", nil, nil, nil],
      13023 => [0, 0, nil, "タ", nil, nil, nil],
      13024 => [0, 0, nil, "チ", nil, nil, nil],
      13025 => [0, 0, nil, "ツ", nil, nil, nil],
      13026 => [0, 0, nil, "テ", nil, nil, nil],
      13027 => [0, 0, nil, "ト", nil, nil, nil],
      13028 => [0, 0, nil, "ナ", nil, nil, nil],
      13029 => [0, 0, nil, "ニ", nil, nil, nil],
      13030 => [0, 0, nil, "ヌ", nil, nil, nil],
      13031 => [0, 0, nil, "ネ", nil, nil, nil],
      13032 => [0, 0, nil, "ノ", nil, nil, nil],
      13033 => [0, 0, nil, "ハ", nil, nil, nil],
      13034 => [0, 0, nil, "ヒ", nil, nil, nil],
      13035 => [0, 0, nil, "フ", nil, nil, nil],
      13036 => [0, 0, nil, "ヘ", nil, nil, nil],
      13037 => [0, 0, nil, "ホ", nil, nil, nil],
      13038 => [0, 0, nil, "マ", nil, nil, nil],
      13039 => [0, 0, nil, "ミ", nil, nil, nil],
      13040 => [0, 0, nil, "ム", nil, nil, nil],
      13041 => [0, 0, nil, "メ", nil, nil, nil],
      13042 => [0, 0, nil, "モ", nil, nil, nil],
      13043 => [0, 0, nil, "ヤ", nil, nil, nil],
      13044 => [0, 0, nil, "ユ", nil, nil, nil],
      13045 => [0, 0, nil, "ヨ", nil, nil, nil],
      13046 => [0, 0, nil, "ラ", nil, nil, nil],
      13047 => [0, 0, nil, "リ", nil, nil, nil],
      13048 => [0, 0, nil, "ル", nil, nil, nil],
      13049 => [0, 0, nil, "レ", nil, nil, nil],
      13050 => [0, 0, nil, "ロ", nil, nil, nil],
      13051 => [0, 0, nil, "ワ", nil, nil, nil],
      13052 => [0, 0, nil, "ヰ", nil, nil, nil],
      13053 => [0, 0, nil, "ヱ", nil, nil, nil],
      13054 => [0, 0, nil, "ヲ", nil, nil, nil],
      13056 => [0, 0, nil, "アパート", nil, nil, nil],
      13057 => [0, 0, nil, "アルファ", nil, nil, nil],
      13058 => [0, 0, nil, "アンペア", nil, nil, nil],
      13059 => [0, 0, nil, "アール", nil, nil, nil],
      13060 => [0, 0, nil, "イニング", nil, nil, nil],
      13061 => [0, 0, nil, "インチ", nil, nil, nil],
      13062 => [0, 0, nil, "ウォン", nil, nil, nil],
      13063 => [0, 0, nil, "エスクード", nil, nil, nil],
      13064 => [0, 0, nil, "エーカー", nil, nil, nil],
      13065 => [0, 0, nil, "オンス", nil, nil, nil],
      13066 => [0, 0, nil, "オーム", nil, nil, nil],
      13067 => [0, 0, nil, "カイリ", nil, nil, nil],
      13068 => [0, 0, nil, "カラット", nil, nil, nil],
      13069 => [0, 0, nil, "カロリー", nil, nil, nil],
      13070 => [0, 0, nil, "ガロン", nil, nil, nil],
      13071 => [0, 0, nil, "ガンマ", nil, nil, nil],
      13072 => [0, 0, nil, "ギガ", nil, nil, nil],
      13073 => [0, 0, nil, "ギニー", nil, nil, nil],
      13074 => [0, 0, nil, "キュリー", nil, nil, nil],
      13075 => [0, 0, nil, "ギルダー", nil, nil, nil],
      13076 => [0, 0, nil, "キロ", nil, nil, nil],
      13077 => [0, 0, nil, "キログラム", nil, nil, nil],
      13078 => [0, 0, nil, "キロメートル", nil, nil, nil],
      13079 => [0, 0, nil, "キロワット", nil, nil, nil],
      13080 => [0, 0, nil, "グラム", nil, nil, nil],
      13081 => [0, 0, nil, "グラムトン", nil, nil, nil],
      13082 => [0, 0, nil, "クルゼイロ", nil, nil, nil],
      13083 => [0, 0, nil, "クローネ", nil, nil, nil],
      13084 => [0, 0, nil, "ケース", nil, nil, nil],
      13085 => [0, 0, nil, "コルナ", nil, nil, nil],
      13086 => [0, 0, nil, "コーポ", nil, nil, nil],
      13087 => [0, 0, nil, "サイクル", nil, nil, nil],
      13088 => [0, 0, nil, "サンチーム", nil, nil, nil],
      13089 => [0, 0, nil, "シリング", nil, nil, nil],
      13090 => [0, 0, nil, "センチ", nil, nil, nil],
      13091 => [0, 0, nil, "セント", nil, nil, nil],
      13092 => [0, 0, nil, "ダース", nil, nil, nil],
      13093 => [0, 0, nil, "デシ", nil, nil, nil],
      13094 => [0, 0, nil, "ドル", nil, nil, nil],
      13095 => [0, 0, nil, "トン", nil, nil, nil],
      13096 => [0, 0, nil, "ナノ", nil, nil, nil],
      13097 => [0, 0, nil, "ノット", nil, nil, nil],
      13098 => [0, 0, nil, "ハイツ", nil, nil, nil],
      13099 => [0, 0, nil, "パーセント", nil, nil, nil],
      13100 => [0, 0, nil, "パーツ", nil, nil, nil],
      13101 => [0, 0, nil, "バーレル", nil, nil, nil],
      13102 => [0, 0, nil, "ピアストル", nil, nil, nil],
      13103 => [0, 0, nil, "ピクル", nil, nil, nil],
      13104 => [0, 0, nil, "ピコ", nil, nil, nil],
      13105 => [0, 0, nil, "ビル", nil, nil, nil],
      13106 => [0, 0, nil, "ファラッド", nil, nil, nil],
      13107 => [0, 0, nil, "フィート", nil, nil, nil],
      13108 => [0, 0, nil, "ブッシェル", nil, nil, nil],
      13109 => [0, 0, nil, "フラン", nil, nil, nil],
      13110 => [0, 0, nil, "ヘクタール", nil, nil, nil],
      13111 => [0, 0, nil, "ペソ", nil, nil, nil],
      13112 => [0, 0, nil, "ペニヒ", nil, nil, nil],
      13113 => [0, 0, nil, "ヘルツ", nil, nil, nil],
      13114 => [0, 0, nil, "ペンス", nil, nil, nil],
      13115 => [0, 0, nil, "ページ", nil, nil, nil],
      13116 => [0, 0, nil, "ベータ", nil, nil, nil],
      13117 => [0, 0, nil, "ポイント", nil, nil, nil],
      13118 => [0, 0, nil, "ボルト", nil, nil, nil],
      13119 => [0, 0, nil, "ホン", nil, nil, nil],
      13120 => [0, 0, nil, "ポンド", nil, nil, nil],
      13121 => [0, 0, nil, "ホール", nil, nil, nil],
      13122 => [0, 0, nil, "ホーン", nil, nil, nil],
      13123 => [0, 0, nil, "マイクロ", nil, nil, nil],
      13124 => [0, 0, nil, "マイル", nil, nil, nil],
      13125 => [0, 0, nil, "マッハ", nil, nil, nil],
      13126 => [0, 0, nil, "マルク", nil, nil, nil],
      13127 => [0, 0, nil, "マンション", nil, nil, nil],
      13128 => [0, 0, nil, "ミクロン", nil, nil, nil],
      13129 => [0, 0, nil, "ミリ", nil, nil, nil],
      13130 => [0, 0, nil, "ミリバール", nil, nil, nil],
      13131 => [0, 0, nil, "メガ", nil, nil, nil],
      13132 => [0, 0, nil, "メガトン", nil, nil, nil],
      13133 => [0, 0, nil, "メートル", nil, nil, nil],
      13134 => [0, 0, nil, "ヤード", nil, nil, nil],
      13135 => [0, 0, nil, "ヤール", nil, nil, nil],
      13136 => [0, 0, nil, "ユアン", nil, nil, nil],
      13137 => [0, 0, nil, "リットル", nil, nil, nil],
      13138 => [0, 0, nil, "リラ", nil, nil, nil],
      13139 => [0, 0, nil, "ルピー", nil, nil, nil],
      13140 => [0, 0, nil, "ルーブル", nil, nil, nil],
      13141 => [0, 0, nil, "レム", nil, nil, nil],
      13142 => [0, 0, nil, "レントゲン", nil, nil, nil],
      13143 => [0, 0, nil, "ワット", nil, nil, nil],
      13144 => [0, 0, nil, "0点", nil, nil, nil],
      13145 => [0, 0, nil, "1点", nil, nil, nil],
      13146 => [0, 0, nil, "2点", nil, nil, nil],
      13147 => [0, 0, nil, "3点", nil, nil, nil],
      13148 => [0, 0, nil, "4点", nil, nil, nil],
      13149 => [0, 0, nil, "5点", nil, nil, nil],
      13150 => [0, 0, nil, "6点", nil, nil, nil],
      13151 => [0, 0, nil, "7点", nil, nil, nil],
      13152 => [0, 0, nil, "8点", nil, nil, nil],
      13153 => [0, 0, nil, "9点", nil, nil, nil],
      13154 => [0, 0, nil, "10点", nil, nil, nil],
      13155 => [0, 0, nil, "11点", nil, nil, nil],
      13156 => [0, 0, nil, "12点", nil, nil, nil],
      13157 => [0, 0, nil, "13点", nil, nil, nil],
      13158 => [0, 0, nil, "14点", nil, nil, nil],
      13159 => [0, 0, nil, "15点", nil, nil, nil],
      13160 => [0, 0, nil, "16点", nil, nil, nil],
      13161 => [0, 0, nil, "17点", nil, nil, nil],
      13162 => [0, 0, nil, "18点", nil, nil, nil],
      13163 => [0, 0, nil, "19点", nil, nil, nil],
      13164 => [0, 0, nil, "20点", nil, nil, nil],
      13165 => [0, 0, nil, "21点", nil, nil, nil],
      13166 => [0, 0, nil, "22点", nil, nil, nil],
      13167 => [0, 0, nil, "23点", nil, nil, nil],
      13168 => [0, 0, nil, "24点", nil, nil, nil],
      13169 => [0, 0, nil, "hPa", nil, nil, nil],
      13170 => [0, 0, nil, "da", nil, nil, nil],
      13171 => [0, 0, nil, "AU", nil, nil, nil],
      13172 => [0, 0, nil, "bar", nil, nil, nil],
      13173 => [0, 0, nil, "oV", nil, nil, nil],
      13174 => [0, 0, nil, "pc", nil, nil, nil],
      13179 => [0, 0, nil, "平成", nil, nil, nil],
      13180 => [0, 0, nil, "昭和", nil, nil, nil],
      13181 => [0, 0, nil, "大正", nil, nil, nil],
      13182 => [0, 0, nil, "明治", nil, nil, nil],
      13183 => [0, 0, nil, "株式会社", nil, nil, nil],
      13184 => [0, 0, nil, "pA", nil, nil, nil],
      13185 => [0, 0, nil, "nA", nil, nil, nil],
      13186 => [0, 0, nil, "μA", nil, nil, nil],
      13187 => [0, 0, nil, "mA", nil, nil, nil],
      13188 => [0, 0, nil, "kA", nil, nil, nil],
      13189 => [0, 0, nil, "KB", nil, nil, nil],
      13190 => [0, 0, nil, "MB", nil, nil, nil],
      13191 => [0, 0, nil, "GB", nil, nil, nil],
      13192 => [0, 0, nil, "cal", nil, nil, nil],
      13193 => [0, 0, nil, "kcal", nil, nil, nil],
      13194 => [0, 0, nil, "pF", nil, nil, nil],
      13195 => [0, 0, nil, "nF", nil, nil, nil],
      13196 => [0, 0, nil, "μF", nil, nil, nil],
      13197 => [0, 0, nil, "μg", nil, nil, nil],
      13198 => [0, 0, nil, "mg", nil, nil, nil],
      13199 => [0, 0, nil, "kg", nil, nil, nil],
      13200 => [0, 0, nil, "Hz", nil, nil, nil],
      13201 => [0, 0, nil, "kHz", nil, nil, nil],
      13202 => [0, 0, nil, "MHz", nil, nil, nil],
      13203 => [0, 0, nil, "GHz", nil, nil, nil],
      13204 => [0, 0, nil, "THz", nil, nil, nil],
      13205 => [0, 0, nil, "μℓ", nil, nil, nil],
      13206 => [0, 0, nil, "mℓ", nil, nil, nil],
      13207 => [0, 0, nil, "dℓ", nil, nil, nil],
      13208 => [0, 0, nil, "kℓ", nil, nil, nil],
      13209 => [0, 0, nil, "fm", nil, nil, nil],
      13210 => [0, 0, nil, "nm", nil, nil, nil],
      13211 => [0, 0, nil, "μm", nil, nil, nil],
      13212 => [0, 0, nil, "mm", nil, nil, nil],
      13213 => [0, 0, nil, "cm", nil, nil, nil],
      13214 => [0, 0, nil, "km", nil, nil, nil],
      13215 => [0, 0, nil, "mm²", nil, nil, nil],
      13216 => [0, 0, nil, "cm²", nil, nil, nil],
      13217 => [0, 0, nil, "m²", nil, nil, nil],
      13218 => [0, 0, nil, "km²", nil, nil, nil],
      13219 => [0, 0, nil, "mm³", nil, nil, nil],
      13220 => [0, 0, nil, "cm³", nil, nil, nil],
      13221 => [0, 0, nil, "m³", nil, nil, nil],
      13222 => [0, 0, nil, "km³", nil, nil, nil],
      13223 => [0, 0, nil, "m∕s", nil, nil, nil],
      13224 => [0, 0, nil, "m∕s²", nil, nil, nil],
      13225 => [0, 0, nil, "Pa", nil, nil, nil],
      13226 => [0, 0, nil, "kPa", nil, nil, nil],
      13227 => [0, 0, nil, "MPa", nil, nil, nil],
      13228 => [0, 0, nil, "GPa", nil, nil, nil],
      13229 => [0, 0, nil, "rad", nil, nil, nil],
      13230 => [0, 0, nil, "rad∕s", nil, nil, nil],
      13231 => [0, 0, nil, "rad∕s²", nil, nil, nil],
      13232 => [0, 0, nil, "ps", nil, nil, nil],
      13233 => [0, 0, nil, "ns", nil, nil, nil],
      13234 => [0, 0, nil, "μs", nil, nil, nil],
      13235 => [0, 0, nil, "ms", nil, nil, nil],
      13236 => [0, 0, nil, "pV", nil, nil, nil],
      13237 => [0, 0, nil, "nV", nil, nil, nil],
      13238 => [0, 0, nil, "μV", nil, nil, nil],
      13239 => [0, 0, nil, "mV", nil, nil, nil],
      13240 => [0, 0, nil, "kV", nil, nil, nil],
      13241 => [0, 0, nil, "MV", nil, nil, nil],
      13242 => [0, 0, nil, "pW", nil, nil, nil],
      13243 => [0, 0, nil, "nW", nil, nil, nil],
      13244 => [0, 0, nil, "μW", nil, nil, nil],
      13245 => [0, 0, nil, "mW", nil, nil, nil],
      13246 => [0, 0, nil, "kW", nil, nil, nil],
      13247 => [0, 0, nil, "MW", nil, nil, nil],
      13248 => [0, 0, nil, "kΩ", nil, nil, nil],
      13249 => [0, 0, nil, "MΩ", nil, nil, nil],
      13250 => [0, 0, nil, "a.m.", nil, nil, nil],
      13251 => [0, 0, nil, "Bq", nil, nil, nil],
      13252 => [0, 0, nil, "cc", nil, nil, nil],
      13253 => [0, 0, nil, "cd", nil, nil, nil],
      13254 => [0, 0, nil, "C∕kg", nil, nil, nil],
      13255 => [0, 0, nil, "Co.", nil, nil, nil],
      13256 => [0, 0, nil, "dB", nil, nil, nil],
      13257 => [0, 0, nil, "Gy", nil, nil, nil],
      13258 => [0, 0, nil, "ha", nil, nil, nil],
      13259 => [0, 0, nil, "HP", nil, nil, nil],
      13260 => [0, 0, nil, "in", nil, nil, nil],
      13261 => [0, 0, nil, "KK", nil, nil, nil],
      13262 => [0, 0, nil, "KM", nil, nil, nil],
      13263 => [0, 0, nil, "kt", nil, nil, nil],
      13264 => [0, 0, nil, "lm", nil, nil, nil],
      13265 => [0, 0, nil, "ln", nil, nil, nil],
      13266 => [0, 0, nil, "log", nil, nil, nil],
      13267 => [0, 0, nil, "lx", nil, nil, nil],
      13268 => [0, 0, nil, "mb", nil, nil, nil],
      13269 => [0, 0, nil, "mil", nil, nil, nil],
      13270 => [0, 0, nil, "mol", nil, nil, nil],
      13271 => [0, 0, nil, "PH", nil, nil, nil],
      13272 => [0, 0, nil, "p.m.", nil, nil, nil],
      13273 => [0, 0, nil, "PPM", nil, nil, nil],
      13274 => [0, 0, nil, "PR", nil, nil, nil],
      13275 => [0, 0, nil, "sr", nil, nil, nil],
      13276 => [0, 0, nil, "Sv", nil, nil, nil],
      13277 => [0, 0, nil, "Wb", nil, nil, nil],
      13280 => [0, 0, nil, "1日", nil, nil, nil],
      13281 => [0, 0, nil, "2日", nil, nil, nil],
      13282 => [0, 0, nil, "3日", nil, nil, nil],
      13283 => [0, 0, nil, "4日", nil, nil, nil],
      13284 => [0, 0, nil, "5日", nil, nil, nil],
      13285 => [0, 0, nil, "6日", nil, nil, nil],
      13286 => [0, 0, nil, "7日", nil, nil, nil],
      13287 => [0, 0, nil, "8日", nil, nil, nil],
      13288 => [0, 0, nil, "9日", nil, nil, nil],
      13289 => [0, 0, nil, "10日", nil, nil, nil],
      13290 => [0, 0, nil, "11日", nil, nil, nil],
      13291 => [0, 0, nil, "12日", nil, nil, nil],
      13292 => [0, 0, nil, "13日", nil, nil, nil],
      13293 => [0, 0, nil, "14日", nil, nil, nil],
      13294 => [0, 0, nil, "15日", nil, nil, nil],
      13295 => [0, 0, nil, "16日", nil, nil, nil],
      13296 => [0, 0, nil, "17日", nil, nil, nil],
      13297 => [0, 0, nil, "18日", nil, nil, nil],
      13298 => [0, 0, nil, "19日", nil, nil, nil],
      13299 => [0, 0, nil, "20日", nil, nil, nil],
      13300 => [0, 0, nil, "21日", nil, nil, nil],
      13301 => [0, 0, nil, "22日", nil, nil, nil],
      13302 => [0, 0, nil, "23日", nil, nil, nil],
      13303 => [0, 0, nil, "24日", nil, nil, nil],
      13304 => [0, 0, nil, "25日", nil, nil, nil],
      13305 => [0, 0, nil, "26日", nil, nil, nil],
      13306 => [0, 0, nil, "27日", nil, nil, nil],
      13307 => [0, 0, nil, "28日", nil, nil, nil],
      13308 => [0, 0, nil, "29日", nil, nil, nil],
      13309 => [0, 0, nil, "30日", nil, nil, nil],
      13310 => [0, 0, nil, "31日", nil, nil, nil],
      63744 => [0, 2, "豈", "豈", nil, nil, nil],
      63745 => [0, 2, "更", "更", nil, nil, nil],
      63746 => [0, 2, "車", "車", nil, nil, nil],
      63747 => [0, 2, "賈", "賈", nil, nil, nil],
      63748 => [0, 2, "滑", "滑", nil, nil, nil],
      63749 => [0, 2, "串", "串", nil, nil, nil],
      63750 => [0, 2, "句", "句", nil, nil, nil],
      63751 => [0, 2, "龜", "龜", nil, nil, nil],
      63752 => [0, 2, "龜", "龜", nil, nil, nil],
      63753 => [0, 2, "契", "契", nil, nil, nil],
      63754 => [0, 2, "金", "金", nil, nil, nil],
      63755 => [0, 2, "喇", "喇", nil, nil, nil],
      63756 => [0, 2, "奈", "奈", nil, nil, nil],
      63757 => [0, 2, "懶", "懶", nil, nil, nil],
      63758 => [0, 2, "癩", "癩", nil, nil, nil],
      63759 => [0, 2, "羅", "羅", nil, nil, nil],
      63760 => [0, 2, "蘿", "蘿", nil, nil, nil],
      63761 => [0, 2, "螺", "螺", nil, nil, nil],
      63762 => [0, 2, "裸", "裸", nil, nil, nil],
      63763 => [0, 2, "邏", "邏", nil, nil, nil],
      63764 => [0, 2, "樂", "樂", nil, nil, nil],
      63765 => [0, 2, "洛", "洛", nil, nil, nil],
      63766 => [0, 2, "烙", "烙", nil, nil, nil],
      63767 => [0, 2, "珞", "珞", nil, nil, nil],
      63768 => [0, 2, "落", "落", nil, nil, nil],
      63769 => [0, 2, "酪", "酪", nil, nil, nil],
      63770 => [0, 2, "駱", "駱", nil, nil, nil],
      63771 => [0, 2, "亂", "亂", nil, nil, nil],
      63772 => [0, 2, "卵", "卵", nil, nil, nil],
      63773 => [0, 2, "欄", "欄", nil, nil, nil],
      63774 => [0, 2, "爛", "爛", nil, nil, nil],
      63775 => [0, 2, "蘭", "蘭", nil, nil, nil],
      63776 => [0, 2, "鸞", "鸞", nil, nil, nil],
      63777 => [0, 2, "嵐", "嵐", nil, nil, nil],
      63778 => [0, 2, "濫", "濫", nil, nil, nil],
      63779 => [0, 2, "藍", "藍", nil, nil, nil],
      63780 => [0, 2, "襤", "襤", nil, nil, nil],
      63781 => [0, 2, "拉", "拉", nil, nil, nil],
      63782 => [0, 2, "臘", "臘", nil, nil, nil],
      63783 => [0, 2, "蠟", "蠟", nil, nil, nil],
      63784 => [0, 2, "廊", "廊", nil, nil, nil],
      63785 => [0, 2, "朗", "朗", nil, nil, nil],
      63786 => [0, 2, "浪", "浪", nil, nil, nil],
      63787 => [0, 2, "狼", "狼", nil, nil, nil],
      63788 => [0, 2, "郎", "郎", nil, nil, nil],
      63789 => [0, 2, "來", "來", nil, nil, nil],
      63790 => [0, 2, "冷", "冷", nil, nil, nil],
      63791 => [0, 2, "勞", "勞", nil, nil, nil],
      63792 => [0, 2, "擄", "擄", nil, nil, nil],
      63793 => [0, 2, "櫓", "櫓", nil, nil, nil],
      63794 => [0, 2, "爐", "爐", nil, nil, nil],
      63795 => [0, 2, "盧", "盧", nil, nil, nil],
      63796 => [0, 2, "老", "老", nil, nil, nil],
      63797 => [0, 2, "蘆", "蘆", nil, nil, nil],
      63798 => [0, 2, "虜", "虜", nil, nil, nil],
      63799 => [0, 2, "路", "路", nil, nil, nil],
      63800 => [0, 2, "露", "露", nil, nil, nil],
      63801 => [0, 2, "魯", "魯", nil, nil, nil],
      63802 => [0, 2, "鷺", "鷺", nil, nil, nil],
      63803 => [0, 2, "碌", "碌", nil, nil, nil],
      63804 => [0, 2, "祿", "祿", nil, nil, nil],
      63805 => [0, 2, "綠", "綠", nil, nil, nil],
      63806 => [0, 2, "菉", "菉", nil, nil, nil],
      63807 => [0, 2, "錄", "錄", nil, nil, nil],
      63808 => [0, 2, "鹿", "鹿", nil, nil, nil],
      63809 => [0, 2, "論", "論", nil, nil, nil],
      63810 => [0, 2, "壟", "壟", nil, nil, nil],
      63811 => [0, 2, "弄", "弄", nil, nil, nil],
      63812 => [0, 2, "籠", "籠", nil, nil, nil],
      63813 => [0, 2, "聾", "聾", nil, nil, nil],
      63814 => [0, 2, "牢", "牢", nil, nil, nil],
      63815 => [0, 2, "磊", "磊", nil, nil, nil],
      63816 => [0, 2, "賂", "賂", nil, nil, nil],
      63817 => [0, 2, "雷", "雷", nil, nil, nil],
      63818 => [0, 2, "壘", "壘", nil, nil, nil],
      63819 => [0, 2, "屢", "屢", nil, nil, nil],
      63820 => [0, 2, "樓", "樓", nil, nil, nil],
      63821 => [0, 2, "淚", "淚", nil, nil, nil],
      63822 => [0, 2, "漏", "漏", nil, nil, nil],
      63823 => [0, 2, "累", "累", nil, nil, nil],
      63824 => [0, 2, "縷", "縷", nil, nil, nil],
      63825 => [0, 2, "電", "電", nil, nil, nil],
      63826 => [0, 2, "勒", "勒", nil, nil, nil],
      63827 => [0, 2, "肋", "肋", nil, nil, nil],
      63828 => [0, 2, "凜", "凜", nil, nil, nil],
      63829 => [0, 2, "凌", "凌", nil, nil, nil],
      63830 => [0, 2, "稜", "稜", nil, nil, nil],
      63831 => [0, 2, "綾", "綾", nil, nil, nil],
      63832 => [0, 2, "菱", "菱", nil, nil, nil],
      63833 => [0, 2, "陵", "陵", nil, nil, nil],
      63834 => [0, 2, "讀", "讀", nil, nil, nil],
      63835 => [0, 2, "拏", "拏", nil, nil, nil],
      63836 => [0, 2, "樂", "樂", nil, nil, nil],
      63837 => [0, 2, "諾", "諾", nil, nil, nil],
      63838 => [0, 2, "丹", "丹", nil, nil, nil],
      63839 => [0, 2, "寧", "寧", nil, nil, nil],
      63840 => [0, 2, "怒", "怒", nil, nil, nil],
      63841 => [0, 2, "率", "率", nil, nil, nil],
      63842 => [0, 2, "異", "異", nil, nil, nil],
      63843 => [0, 2, "北", "北", nil, nil, nil],
      63844 => [0, 2, "磻", "磻", nil, nil, nil],
      63845 => [0, 2, "便", "便", nil, nil, nil],
      63846 => [0, 2, "復", "復", nil, nil, nil],
      63847 => [0, 2, "不", "不", nil, nil, nil],
      63848 => [0, 2, "泌", "泌", nil, nil, nil],
      63849 => [0, 2, "數", "數", nil, nil, nil],
      63850 => [0, 2, "索", "索", nil, nil, nil],
      63851 => [0, 2, "參", "參", nil, nil, nil],
      63852 => [0, 2, "塞", "塞", nil, nil, nil],
      63853 => [0, 2, "省", "省", nil, nil, nil],
      63854 => [0, 2, "葉", "葉", nil, nil, nil],
      63855 => [0, 2, "說", "說", nil, nil, nil],
      63856 => [0, 2, "殺", "殺", nil, nil, nil],
      63857 => [0, 2, "辰", "辰", nil, nil, nil],
      63858 => [0, 2, "沈", "沈", nil, nil, nil],
      63859 => [0, 2, "拾", "拾", nil, nil, nil],
      63860 => [0, 2, "若", "若", nil, nil, nil],
      63861 => [0, 2, "掠", "掠", nil, nil, nil],
      63862 => [0, 2, "略", "略", nil, nil, nil],
      63863 => [0, 2, "亮", "亮", nil, nil, nil],
      63864 => [0, 2, "兩", "兩", nil, nil, nil],
      63865 => [0, 2, "凉", "凉", nil, nil, nil],
      63866 => [0, 2, "梁", "梁", nil, nil, nil],
      63867 => [0, 2, "糧", "糧", nil, nil, nil],
      63868 => [0, 2, "良", "良", nil, nil, nil],
      63869 => [0, 2, "諒", "諒", nil, nil, nil],
      63870 => [0, 2, "量", "量", nil, nil, nil],
      63871 => [0, 2, "勵", "勵", nil, nil, nil],
      63872 => [0, 2, "呂", "呂", nil, nil, nil],
      63873 => [0, 2, "女", "女", nil, nil, nil],
      63874 => [0, 2, "廬", "廬", nil, nil, nil],
      63875 => [0, 2, "旅", "旅", nil, nil, nil],
      63876 => [0, 2, "濾", "濾", nil, nil, nil],
      63877 => [0, 2, "礪", "礪", nil, nil, nil],
      63878 => [0, 2, "閭", "閭", nil, nil, nil],
      63879 => [0, 2, "驪", "驪", nil, nil, nil],
      63880 => [0, 2, "麗", "麗", nil, nil, nil],
      63881 => [0, 2, "黎", "黎", nil, nil, nil],
      63882 => [0, 2, "力", "力", nil, nil, nil],
      63883 => [0, 2, "曆", "曆", nil, nil, nil],
      63884 => [0, 2, "歷", "歷", nil, nil, nil],
      63885 => [0, 2, "轢", "轢", nil, nil, nil],
      63886 => [0, 2, "年", "年", nil, nil, nil],
      63887 => [0, 2, "憐", "憐", nil, nil, nil],
      63888 => [0, 2, "戀", "戀", nil, nil, nil],
      63889 => [0, 2, "撚", "撚", nil, nil, nil],
      63890 => [0, 2, "漣", "漣", nil, nil, nil],
      63891 => [0, 2, "煉", "煉", nil, nil, nil],
      63892 => [0, 2, "璉", "璉", nil, nil, nil],
      63893 => [0, 2, "秊", "秊", nil, nil, nil],
      63894 => [0, 2, "練", "練", nil, nil, nil],
      63895 => [0, 2, "聯", "聯", nil, nil, nil],
      63896 => [0, 2, "輦", "輦", nil, nil, nil],
      63897 => [0, 2, "蓮", "蓮", nil, nil, nil],
      63898 => [0, 2, "連", "連", nil, nil, nil],
      63899 => [0, 2, "鍊", "鍊", nil, nil, nil],
      63900 => [0, 2, "列", "列", nil, nil, nil],
      63901 => [0, 2, "劣", "劣", nil, nil, nil],
      63902 => [0, 2, "咽", "咽", nil, nil, nil],
      63903 => [0, 2, "烈", "烈", nil, nil, nil],
      63904 => [0, 2, "裂", "裂", nil, nil, nil],
      63905 => [0, 2, "說", "說", nil, nil, nil],
      63906 => [0, 2, "廉", "廉", nil, nil, nil],
      63907 => [0, 2, "念", "念", nil, nil, nil],
      63908 => [0, 2, "捻", "捻", nil, nil, nil],
      63909 => [0, 2, "殮", "殮", nil, nil, nil],
      63910 => [0, 2, "簾", "簾", nil, nil, nil],
      63911 => [0, 2, "獵", "獵", nil, nil, nil],
      63912 => [0, 2, "令", "令", nil, nil, nil],
      63913 => [0, 2, "囹", "囹", nil, nil, nil],
      63914 => [0, 2, "寧", "寧", nil, nil, nil],
      63915 => [0, 2, "嶺", "嶺", nil, nil, nil],
      63916 => [0, 2, "怜", "怜", nil, nil, nil],
      63917 => [0, 2, "玲", "玲", nil, nil, nil],
      63918 => [0, 2, "瑩", "瑩", nil, nil, nil],
      63919 => [0, 2, "羚", "羚", nil, nil, nil],
      63920 => [0, 2, "聆", "聆", nil, nil, nil],
      63921 => [0, 2, "鈴", "鈴", nil, nil, nil],
      63922 => [0, 2, "零", "零", nil, nil, nil],
      63923 => [0, 2, "靈", "靈", nil, nil, nil],
      63924 => [0, 2, "領", "領", nil, nil, nil],
      63925 => [0, 2, "例", "例", nil, nil, nil],
      63926 => [0, 2, "禮", "禮", nil, nil, nil],
      63927 => [0, 2, "醴", "醴", nil, nil, nil],
      63928 => [0, 2, "隸", "隸", nil, nil, nil],
      63929 => [0, 2, "惡", "惡", nil, nil, nil],
      63930 => [0, 2, "了", "了", nil, nil, nil],
      63931 => [0, 2, "僚", "僚", nil, nil, nil],
      63932 => [0, 2, "寮", "寮", nil, nil, nil],
      63933 => [0, 2, "尿", "尿", nil, nil, nil],
      63934 => [0, 2, "料", "料", nil, nil, nil],
      63935 => [0, 2, "樂", "樂", nil, nil, nil],
      63936 => [0, 2, "燎", "燎", nil, nil, nil],
      63937 => [0, 2, "療", "療", nil, nil, nil],
      63938 => [0, 2, "蓼", "蓼", nil, nil, nil],
      63939 => [0, 2, "遼", "遼", nil, nil, nil],
      63940 => [0, 2, "龍", "龍", nil, nil, nil],
      63941 => [0, 2, "暈", "暈", nil, nil, nil],
      63942 => [0, 2, "阮", "阮", nil, nil, nil],
      63943 => [0, 2, "劉", "劉", nil, nil, nil],
      63944 => [0, 2, "杻", "杻", nil, nil, nil],
      63945 => [0, 2, "柳", "柳", nil, nil, nil],
      63946 => [0, 2, "流", "流", nil, nil, nil],
      63947 => [0, 2, "溜", "溜", nil, nil, nil],
      63948 => [0, 2, "琉", "琉", nil, nil, nil],
      63949 => [0, 2, "留", "留", nil, nil, nil],
      63950 => [0, 2, "硫", "硫", nil, nil, nil],
      63951 => [0, 2, "紐", "紐", nil, nil, nil],
      63952 => [0, 2, "類", "類", nil, nil, nil],
      63953 => [0, 2, "六", "六", nil, nil, nil],
      63954 => [0, 2, "戮", "戮", nil, nil, nil],
      63955 => [0, 2, "陸", "陸", nil, nil, nil],
      63956 => [0, 2, "倫", "倫", nil, nil, nil],
      63957 => [0, 2, "崙", "崙", nil, nil, nil],
      63958 => [0, 2, "淪", "淪", nil, nil, nil],
      63959 => [0, 2, "輪", "輪", nil, nil, nil],
      63960 => [0, 2, "律", "律", nil, nil, nil],
      63961 => [0, 2, "慄", "慄", nil, nil, nil],
      63962 => [0, 2, "栗", "栗", nil, nil, nil],
      63963 => [0, 2, "率", "率", nil, nil, nil],
      63964 => [0, 2, "隆", "隆", nil, nil, nil],
      63965 => [0, 2, "利", "利", nil, nil, nil],
      63966 => [0, 2, "吏", "吏", nil, nil, nil],
      63967 => [0, 2, "履", "履", nil, nil, nil],
      63968 => [0, 2, "易", "易", nil, nil, nil],
      63969 => [0, 2, "李", "李", nil, nil, nil],
      63970 => [0, 2, "梨", "梨", nil, nil, nil],
      63971 => [0, 2, "泥", "泥", nil, nil, nil],
      63972 => [0, 2, "理", "理", nil, nil, nil],
      63973 => [0, 2, "痢", "痢", nil, nil, nil],
      63974 => [0, 2, "罹", "罹", nil, nil, nil],
      63975 => [0, 2, "裏", "裏", nil, nil, nil],
      63976 => [0, 2, "裡", "裡", nil, nil, nil],
      63977 => [0, 2, "里", "里", nil, nil, nil],
      63978 => [0, 2, "離", "離", nil, nil, nil],
      63979 => [0, 2, "匿", "匿", nil, nil, nil],
      63980 => [0, 2, "溺", "溺", nil, nil, nil],
      63981 => [0, 2, "吝", "吝", nil, nil, nil],
      63982 => [0, 2, "燐", "燐", nil, nil, nil],
      63983 => [0, 2, "璘", "璘", nil, nil, nil],
      63984 => [0, 2, "藺", "藺", nil, nil, nil],
      63985 => [0, 2, "隣", "隣", nil, nil, nil],
      63986 => [0, 2, "鱗", "鱗", nil, nil, nil],
      63987 => [0, 2, "麟", "麟", nil, nil, nil],
      63988 => [0, 2, "林", "林", nil, nil, nil],
      63989 => [0, 2, "淋", "淋", nil, nil, nil],
      63990 => [0, 2, "臨", "臨", nil, nil, nil],
      63991 => [0, 2, "立", "立", nil, nil, nil],
      63992 => [0, 2, "笠", "笠", nil, nil, nil],
      63993 => [0, 2, "粒", "粒", nil, nil, nil],
      63994 => [0, 2, "狀", "狀", nil, nil, nil],
      63995 => [0, 2, "炙", "炙", nil, nil, nil],
      63996 => [0, 2, "識", "識", nil, nil, nil],
      63997 => [0, 2, "什", "什", nil, nil, nil],
      63998 => [0, 2, "茶", "茶", nil, nil, nil],
      63999 => [0, 2, "刺", "刺", nil, nil, nil],
      64000 => [0, 2, "切", "切", nil, nil, nil],
      64001 => [0, 2, "度", "度", nil, nil, nil],
      64002 => [0, 2, "拓", "拓", nil, nil, nil],
      64003 => [0, 2, "糖", "糖", nil, nil, nil],
      64004 => [0, 2, "宅", "宅", nil, nil, nil],
      64005 => [0, 2, "洞", "洞", nil, nil, nil],
      64006 => [0, 2, "暴", "暴", nil, nil, nil],
      64007 => [0, 2, "輻", "輻", nil, nil, nil],
      64008 => [0, 2, "行", "行", nil, nil, nil],
      64009 => [0, 2, "降", "降", nil, nil, nil],
      64010 => [0, 2, "見", "見", nil, nil, nil],
      64011 => [0, 2, "廓", "廓", nil, nil, nil],
      64012 => [0, 2, "兀", "兀", nil, nil, nil],
      64013 => [0, 2, "嗀", "嗀", nil, nil, nil],
      64016 => [0, 2, "塚", "塚", nil, nil, nil],
      64018 => [0, 2, "晴", "晴", nil, nil, nil],
      64021 => [0, 2, "凞", "凞", nil, nil, nil],
      64022 => [0, 2, "猪", "猪", nil, nil, nil],
      64023 => [0, 2, "益", "益", nil, nil, nil],
      64024 => [0, 2, "礼", "礼", nil, nil, nil],
      64025 => [0, 2, "神", "神", nil, nil, nil],
      64026 => [0, 2, "祥", "祥", nil, nil, nil],
      64027 => [0, 2, "福", "福", nil, nil, nil],
      64028 => [0, 2, "靖", "靖", nil, nil, nil],
      64029 => [0, 2, "精", "精", nil, nil, nil],
      64030 => [0, 2, "羽", "羽", nil, nil, nil],
      64032 => [0, 2, "蘒", "蘒", nil, nil, nil],
      64034 => [0, 2, "諸", "諸", nil, nil, nil],
      64037 => [0, 2, "逸", "逸", nil, nil, nil],
      64038 => [0, 2, "都", "都", nil, nil, nil],
      64042 => [0, 2, "飯", "飯", nil, nil, nil],
      64043 => [0, 2, "飼", "飼", nil, nil, nil],
      64044 => [0, 2, "館", "館", nil, nil, nil],
      64045 => [0, 2, "鶴", "鶴", nil, nil, nil],
      64256 => [0, 0, nil, "ff", nil, nil, nil],
      64257 => [0, 0, nil, "fi", nil, nil, nil],
      64258 => [0, 0, nil, "fl", nil, nil, nil],
      64259 => [0, 0, nil, "ffi", nil, nil, nil],
      64260 => [0, 0, nil, "ffl", nil, nil, nil],
      64261 => [0, 0, nil, "ſt", nil, nil, nil],
      64262 => [0, 0, nil, "st", nil, nil, nil],
      64275 => [0, 0, nil, "մն", nil, nil, nil],
      64276 => [0, 0, nil, "մե", nil, nil, nil],
      64277 => [0, 0, nil, "մի", nil, nil, nil],
      64278 => [0, 0, nil, "վն", nil, nil, nil],
      64279 => [0, 0, nil, "մխ", nil, nil, nil],
      64285 => [0, 0, "יִ", "יִ", nil, nil, nil],
      64286 => [26, 0, nil, nil, nil, nil, nil],
      64287 => [0, 1, "ײַ", "ײַ", nil, nil, nil],
      64288 => [0, 0, nil, "ע", nil, nil, nil],
      64289 => [0, 0, nil, "א", nil, nil, nil],
      64290 => [0, 0, nil, "ד", nil, nil, nil],
      64291 => [0, 0, nil, "ה", nil, nil, nil],
      64292 => [0, 0, nil, "כ", nil, nil, nil],
      64293 => [0, 0, nil, "ל", nil, nil, nil],
      64294 => [0, 0, nil, "ם", nil, nil, nil],
      64295 => [0, 0, nil, "ר", nil, nil, nil],
      64296 => [0, 0, nil, "ת", nil, nil, nil],
      64297 => [0, 0, nil, "+", nil, nil, nil],
      64298 => [0, 1, "שׁ", "שׁ", nil, nil, nil],
      64299 => [0, 1, "שׂ", "שׂ", nil, nil, nil],
      64300 => [0, 1, "שּׁ", "שּׁ", nil, nil, nil],
      64301 => [0, 1, "שּׂ", "שּׂ", nil, nil, nil],
      64302 => [0, 1, "אַ", "אַ", nil, nil, nil],
      64303 => [0, 1, "אָ", "אָ", nil, nil, nil],
      64304 => [0, 1, "אּ", "אּ", nil, nil, nil],
      64305 => [0, 1, "בּ", "בּ", nil, nil, nil],
      64306 => [0, 1, "גּ", "גּ", nil, nil, nil],
      64307 => [0, 1, "דּ", "דּ", nil, nil, nil],
      64308 => [0, 1, "הּ", "הּ", nil, nil, nil],
      64309 => [0, 1, "וּ", "וּ", nil, nil, nil],
      64310 => [0, 1, "זּ", "זּ", nil, nil, nil],
      64312 => [0, 1, "טּ", "טּ", nil, nil, nil],
      64313 => [0, 1, "יּ", "יּ", nil, nil, nil],
      64314 => [0, 1, "ךּ", "ךּ", nil, nil, nil],
      64315 => [0, 1, "כּ", "כּ", nil, nil, nil],
      64316 => [0, 1, "לּ", "לּ", nil, nil, nil],
      64318 => [0, 1, "מּ", "מּ", nil, nil, nil],
      64320 => [0, 1, "נּ", "נּ", nil, nil, nil],
      64321 => [0, 1, "סּ", "סּ", nil, nil, nil],
      64323 => [0, 1, "ףּ", "ףּ", nil, nil, nil],
      64324 => [0, 1, "פּ", "פּ", nil, nil, nil],
      64326 => [0, 1, "צּ", "צּ", nil, nil, nil],
      64327 => [0, 1, "קּ", "קּ", nil, nil, nil],
      64328 => [0, 1, "רּ", "רּ", nil, nil, nil],
      64329 => [0, 1, "שּ", "שּ", nil, nil, nil],
      64330 => [0, 1, "תּ", "תּ", nil, nil, nil],
      64331 => [0, 1, "וֹ", "וֹ", nil, nil, nil],
      64332 => [0, 1, "בֿ", "בֿ", nil, nil, nil],
      64333 => [0, 1, "כֿ", "כֿ", nil, nil, nil],
      64334 => [0, 1, "פֿ", "פֿ", nil, nil, nil],
      64335 => [0, 0, nil, "אל", nil, nil, nil],
      64336 => [0, 0, nil, "ٱ", nil, nil, nil],
      64337 => [0, 0, nil, "ٱ", nil, nil, nil],
      64338 => [0, 0, nil, "ٻ", nil, nil, nil],
      64339 => [0, 0, nil, "ٻ", nil, nil, nil],
      64340 => [0, 0, nil, "ٻ", nil, nil, nil],
      64341 => [0, 0, nil, "ٻ", nil, nil, nil],
      64342 => [0, 0, nil, "پ", nil, nil, nil],
      64343 => [0, 0, nil, "پ", nil, nil, nil],
      64344 => [0, 0, nil, "پ", nil, nil, nil],
      64345 => [0, 0, nil, "پ", nil, nil, nil],
      64346 => [0, 0, nil, "ڀ", nil, nil, nil],
      64347 => [0, 0, nil, "ڀ", nil, nil, nil],
      64348 => [0, 0, nil, "ڀ", nil, nil, nil],
      64349 => [0, 0, nil, "ڀ", nil, nil, nil],
      64350 => [0, 0, nil, "ٺ", nil, nil, nil],
      64351 => [0, 0, nil, "ٺ", nil, nil, nil],
      64352 => [0, 0, nil, "ٺ", nil, nil, nil],
      64353 => [0, 0, nil, "ٺ", nil, nil, nil],
      64354 => [0, 0, nil, "ٿ", nil, nil, nil],
      64355 => [0, 0, nil, "ٿ", nil, nil, nil],
      64356 => [0, 0, nil, "ٿ", nil, nil, nil],
      64357 => [0, 0, nil, "ٿ", nil, nil, nil],
      64358 => [0, 0, nil, "ٹ", nil, nil, nil],
      64359 => [0, 0, nil, "ٹ", nil, nil, nil],
      64360 => [0, 0, nil, "ٹ", nil, nil, nil],
      64361 => [0, 0, nil, "ٹ", nil, nil, nil],
      64362 => [0, 0, nil, "ڤ", nil, nil, nil],
      64363 => [0, 0, nil, "ڤ", nil, nil, nil],
      64364 => [0, 0, nil, "ڤ", nil, nil, nil],
      64365 => [0, 0, nil, "ڤ", nil, nil, nil],
      64366 => [0, 0, nil, "ڦ", nil, nil, nil],
      64367 => [0, 0, nil, "ڦ", nil, nil, nil],
      64368 => [0, 0, nil, "ڦ", nil, nil, nil],
      64369 => [0, 0, nil, "ڦ", nil, nil, nil],
      64370 => [0, 0, nil, "ڄ", nil, nil, nil],
      64371 => [0, 0, nil, "ڄ", nil, nil, nil],
      64372 => [0, 0, nil, "ڄ", nil, nil, nil],
      64373 => [0, 0, nil, "ڄ", nil, nil, nil],
      64374 => [0, 0, nil, "ڃ", nil, nil, nil],
      64375 => [0, 0, nil, "ڃ", nil, nil, nil],
      64376 => [0, 0, nil, "ڃ", nil, nil, nil],
      64377 => [0, 0, nil, "ڃ", nil, nil, nil],
      64378 => [0, 0, nil, "چ", nil, nil, nil],
      64379 => [0, 0, nil, "چ", nil, nil, nil],
      64380 => [0, 0, nil, "چ", nil, nil, nil],
      64381 => [0, 0, nil, "چ", nil, nil, nil],
      64382 => [0, 0, nil, "ڇ", nil, nil, nil],
      64383 => [0, 0, nil, "ڇ", nil, nil, nil],
      64384 => [0, 0, nil, "ڇ", nil, nil, nil],
      64385 => [0, 0, nil, "ڇ", nil, nil, nil],
      64386 => [0, 0, nil, "ڍ", nil, nil, nil],
      64387 => [0, 0, nil, "ڍ", nil, nil, nil],
      64388 => [0, 0, nil, "ڌ", nil, nil, nil],
      64389 => [0, 0, nil, "ڌ", nil, nil, nil],
      64390 => [0, 0, nil, "ڎ", nil, nil, nil],
      64391 => [0, 0, nil, "ڎ", nil, nil, nil],
      64392 => [0, 0, nil, "ڈ", nil, nil, nil],
      64393 => [0, 0, nil, "ڈ", nil, nil, nil],
      64394 => [0, 0, nil, "ژ", nil, nil, nil],
      64395 => [0, 0, nil, "ژ", nil, nil, nil],
      64396 => [0, 0, nil, "ڑ", nil, nil, nil],
      64397 => [0, 0, nil, "ڑ", nil, nil, nil],
      64398 => [0, 0, nil, "ک", nil, nil, nil],
      64399 => [0, 0, nil, "ک", nil, nil, nil],
      64400 => [0, 0, nil, "ک", nil, nil, nil],
      64401 => [0, 0, nil, "ک", nil, nil, nil],
      64402 => [0, 0, nil, "گ", nil, nil, nil],
      64403 => [0, 0, nil, "گ", nil, nil, nil],
      64404 => [0, 0, nil, "گ", nil, nil, nil],
      64405 => [0, 0, nil, "گ", nil, nil, nil],
      64406 => [0, 0, nil, "ڳ", nil, nil, nil],
      64407 => [0, 0, nil, "ڳ", nil, nil, nil],
      64408 => [0, 0, nil, "ڳ", nil, nil, nil],
      64409 => [0, 0, nil, "ڳ", nil, nil, nil],
      64410 => [0, 0, nil, "ڱ", nil, nil, nil],
      64411 => [0, 0, nil, "ڱ", nil, nil, nil],
      64412 => [0, 0, nil, "ڱ", nil, nil, nil],
      64413 => [0, 0, nil, "ڱ", nil, nil, nil],
      64414 => [0, 0, nil, "ں", nil, nil, nil],
      64415 => [0, 0, nil, "ں", nil, nil, nil],
      64416 => [0, 0, nil, "ڻ", nil, nil, nil],
      64417 => [0, 0, nil, "ڻ", nil, nil, nil],
      64418 => [0, 0, nil, "ڻ", nil, nil, nil],
      64419 => [0, 0, nil, "ڻ", nil, nil, nil],
      64420 => [0, 0, nil, "ۀ", nil, nil, nil],
      64421 => [0, 0, nil, "ۀ", nil, nil, nil],
      64422 => [0, 0, nil, "ہ", nil, nil, nil],
      64423 => [0, 0, nil, "ہ", nil, nil, nil],
      64424 => [0, 0, nil, "ہ", nil, nil, nil],
      64425 => [0, 0, nil, "ہ", nil, nil, nil],
      64426 => [0, 0, nil, "ھ", nil, nil, nil],
      64427 => [0, 0, nil, "ھ", nil, nil, nil],
      64428 => [0, 0, nil, "ھ", nil, nil, nil],
      64429 => [0, 0, nil, "ھ", nil, nil, nil],
      64430 => [0, 0, nil, "ے", nil, nil, nil],
      64431 => [0, 0, nil, "ے", nil, nil, nil],
      64432 => [0, 0, nil, "ۓ", nil, nil, nil],
      64433 => [0, 0, nil, "ۓ", nil, nil, nil],
      64467 => [0, 0, nil, "ڭ", nil, nil, nil],
      64468 => [0, 0, nil, "ڭ", nil, nil, nil],
      64469 => [0, 0, nil, "ڭ", nil, nil, nil],
      64470 => [0, 0, nil, "ڭ", nil, nil, nil],
      64471 => [0, 0, nil, "ۇ", nil, nil, nil],
      64472 => [0, 0, nil, "ۇ", nil, nil, nil],
      64473 => [0, 0, nil, "ۆ", nil, nil, nil],
      64474 => [0, 0, nil, "ۆ", nil, nil, nil],
      64475 => [0, 0, nil, "ۈ", nil, nil, nil],
      64476 => [0, 0, nil, "ۈ", nil, nil, nil],
      64477 => [0, 0, nil, "ٷ", nil, nil, nil],
      64478 => [0, 0, nil, "ۋ", nil, nil, nil],
      64479 => [0, 0, nil, "ۋ", nil, nil, nil],
      64480 => [0, 0, nil, "ۅ", nil, nil, nil],
      64481 => [0, 0, nil, "ۅ", nil, nil, nil],
      64482 => [0, 0, nil, "ۉ", nil, nil, nil],
      64483 => [0, 0, nil, "ۉ", nil, nil, nil],
      64484 => [0, 0, nil, "ې", nil, nil, nil],
      64485 => [0, 0, nil, "ې", nil, nil, nil],
      64486 => [0, 0, nil, "ې", nil, nil, nil],
      64487 => [0, 0, nil, "ې", nil, nil, nil],
      64488 => [0, 0, nil, "ى", nil, nil, nil],
      64489 => [0, 0, nil, "ى", nil, nil, nil],
      64490 => [0, 0, nil, "ئا", nil, nil, nil],
      64491 => [0, 0, nil, "ئا", nil, nil, nil],
      64492 => [0, 0, nil, "ئە", nil, nil, nil],
      64493 => [0, 0, nil, "ئە", nil, nil, nil],
      64494 => [0, 0, nil, "ئو", nil, nil, nil],
      64495 => [0, 0, nil, "ئو", nil, nil, nil],
      64496 => [0, 0, nil, "ئۇ", nil, nil, nil],
      64497 => [0, 0, nil, "ئۇ", nil, nil, nil],
      64498 => [0, 0, nil, "ئۆ", nil, nil, nil],
      64499 => [0, 0, nil, "ئۆ", nil, nil, nil],
      64500 => [0, 0, nil, "ئۈ", nil, nil, nil],
      64501 => [0, 0, nil, "ئۈ", nil, nil, nil],
      64502 => [0, 0, nil, "ئې", nil, nil, nil],
      64503 => [0, 0, nil, "ئې", nil, nil, nil],
      64504 => [0, 0, nil, "ئې", nil, nil, nil],
      64505 => [0, 0, nil, "ئى", nil, nil, nil],
      64506 => [0, 0, nil, "ئى", nil, nil, nil],
      64507 => [0, 0, nil, "ئى", nil, nil, nil],
      64508 => [0, 0, nil, "ی", nil, nil, nil],
      64509 => [0, 0, nil, "ی", nil, nil, nil],
      64510 => [0, 0, nil, "ی", nil, nil, nil],
      64511 => [0, 0, nil, "ی", nil, nil, nil],
      64512 => [0, 0, nil, "ئج", nil, nil, nil],
      64513 => [0, 0, nil, "ئح", nil, nil, nil],
      64514 => [0, 0, nil, "ئم", nil, nil, nil],
      64515 => [0, 0, nil, "ئى", nil, nil, nil],
      64516 => [0, 0, nil, "ئي", nil, nil, nil],
      64517 => [0, 0, nil, "بج", nil, nil, nil],
      64518 => [0, 0, nil, "بح", nil, nil, nil],
      64519 => [0, 0, nil, "بخ", nil, nil, nil],
      64520 => [0, 0, nil, "بم", nil, nil, nil],
      64521 => [0, 0, nil, "بى", nil, nil, nil],
      64522 => [0, 0, nil, "بي", nil, nil, nil],
      64523 => [0, 0, nil, "تج", nil, nil, nil],
      64524 => [0, 0, nil, "تح", nil, nil, nil],
      64525 => [0, 0, nil, "تخ", nil, nil, nil],
      64526 => [0, 0, nil, "تم", nil, nil, nil],
      64527 => [0, 0, nil, "تى", nil, nil, nil],
      64528 => [0, 0, nil, "تي", nil, nil, nil],
      64529 => [0, 0, nil, "ثج", nil, nil, nil],
      64530 => [0, 0, nil, "ثم", nil, nil, nil],
      64531 => [0, 0, nil, "ثى", nil, nil, nil],
      64532 => [0, 0, nil, "ثي", nil, nil, nil],
      64533 => [0, 0, nil, "جح", nil, nil, nil],
      64534 => [0, 0, nil, "جم", nil, nil, nil],
      64535 => [0, 0, nil, "حج", nil, nil, nil],
      64536 => [0, 0, nil, "حم", nil, nil, nil],
      64537 => [0, 0, nil, "خج", nil, nil, nil],
      64538 => [0, 0, nil, "خح", nil, nil, nil],
      64539 => [0, 0, nil, "خم", nil, nil, nil],
      64540 => [0, 0, nil, "سج", nil, nil, nil],
      64541 => [0, 0, nil, "سح", nil, nil, nil],
      64542 => [0, 0, nil, "سخ", nil, nil, nil],
      64543 => [0, 0, nil, "سم", nil, nil, nil],
      64544 => [0, 0, nil, "صح", nil, nil, nil],
      64545 => [0, 0, nil, "صم", nil, nil, nil],
      64546 => [0, 0, nil, "ضج", nil, nil, nil],
      64547 => [0, 0, nil, "ضح", nil, nil, nil],
      64548 => [0, 0, nil, "ضخ", nil, nil, nil],
      64549 => [0, 0, nil, "ضم", nil, nil, nil],
      64550 => [0, 0, nil, "طح", nil, nil, nil],
      64551 => [0, 0, nil, "طم", nil, nil, nil],
      64552 => [0, 0, nil, "ظم", nil, nil, nil],
      64553 => [0, 0, nil, "عج", nil, nil, nil],
      64554 => [0, 0, nil, "عم", nil, nil, nil],
      64555 => [0, 0, nil, "غج", nil, nil, nil],
      64556 => [0, 0, nil, "غم", nil, nil, nil],
      64557 => [0, 0, nil, "فج", nil, nil, nil],
      64558 => [0, 0, nil, "فح", nil, nil, nil],
      64559 => [0, 0, nil, "فخ", nil, nil, nil],
      64560 => [0, 0, nil, "فم", nil, nil, nil],
      64561 => [0, 0, nil, "فى", nil, nil, nil],
      64562 => [0, 0, nil, "في", nil, nil, nil],
      64563 => [0, 0, nil, "قح", nil, nil, nil],
      64564 => [0, 0, nil, "قم", nil, nil, nil],
      64565 => [0, 0, nil, "قى", nil, nil, nil],
      64566 => [0, 0, nil, "قي", nil, nil, nil],
      64567 => [0, 0, nil, "كا", nil, nil, nil],
      64568 => [0, 0, nil, "كج", nil, nil, nil],
      64569 => [0, 0, nil, "كح", nil, nil, nil],
      64570 => [0, 0, nil, "كخ", nil, nil, nil],
      64571 => [0, 0, nil, "كل", nil, nil, nil],
      64572 => [0, 0, nil, "كم", nil, nil, nil],
      64573 => [0, 0, nil, "كى", nil, nil, nil],
      64574 => [0, 0, nil, "كي", nil, nil, nil],
      64575 => [0, 0, nil, "لج", nil, nil, nil],
      64576 => [0, 0, nil, "لح", nil, nil, nil],
      64577 => [0, 0, nil, "لخ", nil, nil, nil],
      64578 => [0, 0, nil, "لم", nil, nil, nil],
      64579 => [0, 0, nil, "لى", nil, nil, nil],
      64580 => [0, 0, nil, "لي", nil, nil, nil],
      64581 => [0, 0, nil, "مج", nil, nil, nil],
      64582 => [0, 0, nil, "مح", nil, nil, nil],
      64583 => [0, 0, nil, "مخ", nil, nil, nil],
      64584 => [0, 0, nil, "مم", nil, nil, nil],
      64585 => [0, 0, nil, "مى", nil, nil, nil],
      64586 => [0, 0, nil, "مي", nil, nil, nil],
      64587 => [0, 0, nil, "نج", nil, nil, nil],
      64588 => [0, 0, nil, "نح", nil, nil, nil],
      64589 => [0, 0, nil, "نخ", nil, nil, nil],
      64590 => [0, 0, nil, "نم", nil, nil, nil],
      64591 => [0, 0, nil, "نى", nil, nil, nil],
      64592 => [0, 0, nil, "ني", nil, nil, nil],
      64593 => [0, 0, nil, "هج", nil, nil, nil],
      64594 => [0, 0, nil, "هم", nil, nil, nil],
      64595 => [0, 0, nil, "هى", nil, nil, nil],
      64596 => [0, 0, nil, "هي", nil, nil, nil],
      64597 => [0, 0, nil, "يج", nil, nil, nil],
      64598 => [0, 0, nil, "يح", nil, nil, nil],
      64599 => [0, 0, nil, "يخ", nil, nil, nil],
      64600 => [0, 0, nil, "يم", nil, nil, nil],
      64601 => [0, 0, nil, "يى", nil, nil, nil],
      64602 => [0, 0, nil, "يي", nil, nil, nil],
      64603 => [0, 0, nil, "ذٰ", nil, nil, nil],
      64604 => [0, 0, nil, "رٰ", nil, nil, nil],
      64605 => [0, 0, nil, "ىٰ", nil, nil, nil],
      64606 => [0, 0, nil, " ٌّ", nil, nil, nil],
      64607 => [0, 0, nil, " ٍّ", nil, nil, nil],
      64608 => [0, 0, nil, " َّ", nil, nil, nil],
      64609 => [0, 0, nil, " ُّ", nil, nil, nil],
      64610 => [0, 0, nil, " ِّ", nil, nil, nil],
      64611 => [0, 0, nil, " ّٰ", nil, nil, nil],
      64612 => [0, 0, nil, "ئر", nil, nil, nil],
      64613 => [0, 0, nil, "ئز", nil, nil, nil],
      64614 => [0, 0, nil, "ئم", nil, nil, nil],
      64615 => [0, 0, nil, "ئن", nil, nil, nil],
      64616 => [0, 0, nil, "ئى", nil, nil, nil],
      64617 => [0, 0, nil, "ئي", nil, nil, nil],
      64618 => [0, 0, nil, "بر", nil, nil, nil],
      64619 => [0, 0, nil, "بز", nil, nil, nil],
      64620 => [0, 0, nil, "بم", nil, nil, nil],
      64621 => [0, 0, nil, "بن", nil, nil, nil],
      64622 => [0, 0, nil, "بى", nil, nil, nil],
      64623 => [0, 0, nil, "بي", nil, nil, nil],
      64624 => [0, 0, nil, "تر", nil, nil, nil],
      64625 => [0, 0, nil, "تز", nil, nil, nil],
      64626 => [0, 0, nil, "تم", nil, nil, nil],
      64627 => [0, 0, nil, "تن", nil, nil, nil],
      64628 => [0, 0, nil, "تى", nil, nil, nil],
      64629 => [0, 0, nil, "تي", nil, nil, nil],
      64630 => [0, 0, nil, "ثر", nil, nil, nil],
      64631 => [0, 0, nil, "ثز", nil, nil, nil],
      64632 => [0, 0, nil, "ثم", nil, nil, nil],
      64633 => [0, 0, nil, "ثن", nil, nil, nil],
      64634 => [0, 0, nil, "ثى", nil, nil, nil],
      64635 => [0, 0, nil, "ثي", nil, nil, nil],
      64636 => [0, 0, nil, "فى", nil, nil, nil],
      64637 => [0, 0, nil, "في", nil, nil, nil],
      64638 => [0, 0, nil, "قى", nil, nil, nil],
      64639 => [0, 0, nil, "قي", nil, nil, nil],
      64640 => [0, 0, nil, "كا", nil, nil, nil],
      64641 => [0, 0, nil, "كل", nil, nil, nil],
      64642 => [0, 0, nil, "كم", nil, nil, nil],
      64643 => [0, 0, nil, "كى", nil, nil, nil],
      64644 => [0, 0, nil, "كي", nil, nil, nil],
      64645 => [0, 0, nil, "لم", nil, nil, nil],
      64646 => [0, 0, nil, "لى", nil, nil, nil],
      64647 => [0, 0, nil, "لي", nil, nil, nil],
      64648 => [0, 0, nil, "ما", nil, nil, nil],
      64649 => [0, 0, nil, "مم", nil, nil, nil],
      64650 => [0, 0, nil, "نر", nil, nil, nil],
      64651 => [0, 0, nil, "نز", nil, nil, nil],
      64652 => [0, 0, nil, "نم", nil, nil, nil],
      64653 => [0, 0, nil, "نن", nil, nil, nil],
      64654 => [0, 0, nil, "نى", nil, nil, nil],
      64655 => [0, 0, nil, "ني", nil, nil, nil],
      64656 => [0, 0, nil, "ىٰ", nil, nil, nil],
      64657 => [0, 0, nil, "ير", nil, nil, nil],
      64658 => [0, 0, nil, "يز", nil, nil, nil],
      64659 => [0, 0, nil, "يم", nil, nil, nil],
      64660 => [0, 0, nil, "ين", nil, nil, nil],
      64661 => [0, 0, nil, "يى", nil, nil, nil],
      64662 => [0, 0, nil, "يي", nil, nil, nil],
      64663 => [0, 0, nil, "ئج", nil, nil, nil],
      64664 => [0, 0, nil, "ئح", nil, nil, nil],
      64665 => [0, 0, nil, "ئخ", nil, nil, nil],
      64666 => [0, 0, nil, "ئم", nil, nil, nil],
      64667 => [0, 0, nil, "ئه", nil, nil, nil],
      64668 => [0, 0, nil, "بج", nil, nil, nil],
      64669 => [0, 0, nil, "بح", nil, nil, nil],
      64670 => [0, 0, nil, "بخ", nil, nil, nil],
      64671 => [0, 0, nil, "بم", nil, nil, nil],
      64672 => [0, 0, nil, "به", nil, nil, nil],
      64673 => [0, 0, nil, "تج", nil, nil, nil],
      64674 => [0, 0, nil, "تح", nil, nil, nil],
      64675 => [0, 0, nil, "تخ", nil, nil, nil],
      64676 => [0, 0, nil, "تم", nil, nil, nil],
      64677 => [0, 0, nil, "ته", nil, nil, nil],
      64678 => [0, 0, nil, "ثم", nil, nil, nil],
      64679 => [0, 0, nil, "جح", nil, nil, nil],
      64680 => [0, 0, nil, "جم", nil, nil, nil],
      64681 => [0, 0, nil, "حج", nil, nil, nil],
      64682 => [0, 0, nil, "حم", nil, nil, nil],
      64683 => [0, 0, nil, "خج", nil, nil, nil],
      64684 => [0, 0, nil, "خم", nil, nil, nil],
      64685 => [0, 0, nil, "سج", nil, nil, nil],
      64686 => [0, 0, nil, "سح", nil, nil, nil],
      64687 => [0, 0, nil, "سخ", nil, nil, nil],
      64688 => [0, 0, nil, "سم", nil, nil, nil],
      64689 => [0, 0, nil, "صح", nil, nil, nil],
      64690 => [0, 0, nil, "صخ", nil, nil, nil],
      64691 => [0, 0, nil, "صم", nil, nil, nil],
      64692 => [0, 0, nil, "ضج", nil, nil, nil],
      64693 => [0, 0, nil, "ضح", nil, nil, nil],
      64694 => [0, 0, nil, "ضخ", nil, nil, nil],
      64695 => [0, 0, nil, "ضم", nil, nil, nil],
      64696 => [0, 0, nil, "طح", nil, nil, nil],
      64697 => [0, 0, nil, "ظم", nil, nil, nil],
      64698 => [0, 0, nil, "عج", nil, nil, nil],
      64699 => [0, 0, nil, "عم", nil, nil, nil],
      64700 => [0, 0, nil, "غج", nil, nil, nil],
      64701 => [0, 0, nil, "غم", nil, nil, nil],
      64702 => [0, 0, nil, "فج", nil, nil, nil],
      64703 => [0, 0, nil, "فح", nil, nil, nil],
      64704 => [0, 0, nil, "فخ", nil, nil, nil],
      64705 => [0, 0, nil, "فم", nil, nil, nil],
      64706 => [0, 0, nil, "قح", nil, nil, nil],
      64707 => [0, 0, nil, "قم", nil, nil, nil],
      64708 => [0, 0, nil, "كج", nil, nil, nil],
      64709 => [0, 0, nil, "كح", nil, nil, nil],
      64710 => [0, 0, nil, "كخ", nil, nil, nil],
      64711 => [0, 0, nil, "كل", nil, nil, nil],
      64712 => [0, 0, nil, "كم", nil, nil, nil],
      64713 => [0, 0, nil, "لج", nil, nil, nil],
      64714 => [0, 0, nil, "لح", nil, nil, nil],
      64715 => [0, 0, nil, "لخ", nil, nil, nil],
      64716 => [0, 0, nil, "لم", nil, nil, nil],
      64717 => [0, 0, nil, "له", nil, nil, nil],
      64718 => [0, 0, nil, "مج", nil, nil, nil],
      64719 => [0, 0, nil, "مح", nil, nil, nil],
      64720 => [0, 0, nil, "مخ", nil, nil, nil],
      64721 => [0, 0, nil, "مم", nil, nil, nil],
      64722 => [0, 0, nil, "نج", nil, nil, nil],
      64723 => [0, 0, nil, "نح", nil, nil, nil],
      64724 => [0, 0, nil, "نخ", nil, nil, nil],
      64725 => [0, 0, nil, "نم", nil, nil, nil],
      64726 => [0, 0, nil, "نه", nil, nil, nil],
      64727 => [0, 0, nil, "هج", nil, nil, nil],
      64728 => [0, 0, nil, "هم", nil, nil, nil],
      64729 => [0, 0, nil, "هٰ", nil, nil, nil],
      64730 => [0, 0, nil, "يج", nil, nil, nil],
      64731 => [0, 0, nil, "يح", nil, nil, nil],
      64732 => [0, 0, nil, "يخ", nil, nil, nil],
      64733 => [0, 0, nil, "يم", nil, nil, nil],
      64734 => [0, 0, nil, "يه", nil, nil, nil],
      64735 => [0, 0, nil, "ئم", nil, nil, nil],
      64736 => [0, 0, nil, "ئه", nil, nil, nil],
      64737 => [0, 0, nil, "بم", nil, nil, nil],
      64738 => [0, 0, nil, "به", nil, nil, nil],
      64739 => [0, 0, nil, "تم", nil, nil, nil],
      64740 => [0, 0, nil, "ته", nil, nil, nil],
      64741 => [0, 0, nil, "ثم", nil, nil, nil],
      64742 => [0, 0, nil, "ثه", nil, nil, nil],
      64743 => [0, 0, nil, "سم", nil, nil, nil],
      64744 => [0, 0, nil, "سه", nil, nil, nil],
      64745 => [0, 0, nil, "شم", nil, nil, nil],
      64746 => [0, 0, nil, "شه", nil, nil, nil],
      64747 => [0, 0, nil, "كل", nil, nil, nil],
      64748 => [0, 0, nil, "كم", nil, nil, nil],
      64749 => [0, 0, nil, "لم", nil, nil, nil],
      64750 => [0, 0, nil, "نم", nil, nil, nil],
      64751 => [0, 0, nil, "نه", nil, nil, nil],
      64752 => [0, 0, nil, "يم", nil, nil, nil],
      64753 => [0, 0, nil, "يه", nil, nil, nil],
      64754 => [0, 0, nil, "ـَّ", nil, nil, nil],
      64755 => [0, 0, nil, "ـُّ", nil, nil, nil],
      64756 => [0, 0, nil, "ـِّ", nil, nil, nil],
      64757 => [0, 0, nil, "طى", nil, nil, nil],
      64758 => [0, 0, nil, "طي", nil, nil, nil],
      64759 => [0, 0, nil, "عى", nil, nil, nil],
      64760 => [0, 0, nil, "عي", nil, nil, nil],
      64761 => [0, 0, nil, "غى", nil, nil, nil],
      64762 => [0, 0, nil, "غي", nil, nil, nil],
      64763 => [0, 0, nil, "سى", nil, nil, nil],
      64764 => [0, 0, nil, "سي", nil, nil, nil],
      64765 => [0, 0, nil, "شى", nil, nil, nil],
      64766 => [0, 0, nil, "شي", nil, nil, nil],
      64767 => [0, 0, nil, "حى", nil, nil, nil],
      64768 => [0, 0, nil, "حي", nil, nil, nil],
      64769 => [0, 0, nil, "جى", nil, nil, nil],
      64770 => [0, 0, nil, "جي", nil, nil, nil],
      64771 => [0, 0, nil, "خى", nil, nil, nil],
      64772 => [0, 0, nil, "خي", nil, nil, nil],
      64773 => [0, 0, nil, "صى", nil, nil, nil],
      64774 => [0, 0, nil, "صي", nil, nil, nil],
      64775 => [0, 0, nil, "ضى", nil, nil, nil],
      64776 => [0, 0, nil, "ضي", nil, nil, nil],
      64777 => [0, 0, nil, "شج", nil, nil, nil],
      64778 => [0, 0, nil, "شح", nil, nil, nil],
      64779 => [0, 0, nil, "شخ", nil, nil, nil],
      64780 => [0, 0, nil, "شم", nil, nil, nil],
      64781 => [0, 0, nil, "شر", nil, nil, nil],
      64782 => [0, 0, nil, "سر", nil, nil, nil],
      64783 => [0, 0, nil, "صر", nil, nil, nil],
      64784 => [0, 0, nil, "ضر", nil, nil, nil],
      64785 => [0, 0, nil, "طى", nil, nil, nil],
      64786 => [0, 0, nil, "طي", nil, nil, nil],
      64787 => [0, 0, nil, "عى", nil, nil, nil],
      64788 => [0, 0, nil, "عي", nil, nil, nil],
      64789 => [0, 0, nil, "غى", nil, nil, nil],
      64790 => [0, 0, nil, "غي", nil, nil, nil],
      64791 => [0, 0, nil, "سى", nil, nil, nil],
      64792 => [0, 0, nil, "سي", nil, nil, nil],
      64793 => [0, 0, nil, "شى", nil, nil, nil],
      64794 => [0, 0, nil, "شي", nil, nil, nil],
      64795 => [0, 0, nil, "حى", nil, nil, nil],
      64796 => [0, 0, nil, "حي", nil, nil, nil],
      64797 => [0, 0, nil, "جى", nil, nil, nil],
      64798 => [0, 0, nil, "جي", nil, nil, nil],
      64799 => [0, 0, nil, "خى", nil, nil, nil],
      64800 => [0, 0, nil, "خي", nil, nil, nil],
      64801 => [0, 0, nil, "صى", nil, nil, nil],
      64802 => [0, 0, nil, "صي", nil, nil, nil],
      64803 => [0, 0, nil, "ضى", nil, nil, nil],
      64804 => [0, 0, nil, "ضي", nil, nil, nil],
      64805 => [0, 0, nil, "شج", nil, nil, nil],
      64806 => [0, 0, nil, "شح", nil, nil, nil],
      64807 => [0, 0, nil, "شخ", nil, nil, nil],
      64808 => [0, 0, nil, "شم", nil, nil, nil],
      64809 => [0, 0, nil, "شر", nil, nil, nil],
      64810 => [0, 0, nil, "سر", nil, nil, nil],
      64811 => [0, 0, nil, "صر", nil, nil, nil],
      64812 => [0, 0, nil, "ضر", nil, nil, nil],
      64813 => [0, 0, nil, "شج", nil, nil, nil],
      64814 => [0, 0, nil, "شح", nil, nil, nil],
      64815 => [0, 0, nil, "شخ", nil, nil, nil],
      64816 => [0, 0, nil, "شم", nil, nil, nil],
      64817 => [0, 0, nil, "سه", nil, nil, nil],
      64818 => [0, 0, nil, "شه", nil, nil, nil],
      64819 => [0, 0, nil, "طم", nil, nil, nil],
      64820 => [0, 0, nil, "سج", nil, nil, nil],
      64821 => [0, 0, nil, "سح", nil, nil, nil],
      64822 => [0, 0, nil, "سخ", nil, nil, nil],
      64823 => [0, 0, nil, "شج", nil, nil, nil],
      64824 => [0, 0, nil, "شح", nil, nil, nil],
      64825 => [0, 0, nil, "شخ", nil, nil, nil],
      64826 => [0, 0, nil, "طم", nil, nil, nil],
      64827 => [0, 0, nil, "ظم", nil, nil, nil],
      64828 => [0, 0, nil, "اً", nil, nil, nil],
      64829 => [0, 0, nil, "اً", nil, nil, nil],
      64848 => [0, 0, nil, "تجم", nil, nil, nil],
      64849 => [0, 0, nil, "تحج", nil, nil, nil],
      64850 => [0, 0, nil, "تحج", nil, nil, nil],
      64851 => [0, 0, nil, "تحم", nil, nil, nil],
      64852 => [0, 0, nil, "تخم", nil, nil, nil],
      64853 => [0, 0, nil, "تمج", nil, nil, nil],
      64854 => [0, 0, nil, "تمح", nil, nil, nil],
      64855 => [0, 0, nil, "تمخ", nil, nil, nil],
      64856 => [0, 0, nil, "جمح", nil, nil, nil],
      64857 => [0, 0, nil, "جمح", nil, nil, nil],
      64858 => [0, 0, nil, "حمي", nil, nil, nil],
      64859 => [0, 0, nil, "حمى", nil, nil, nil],
      64860 => [0, 0, nil, "سحج", nil, nil, nil],
      64861 => [0, 0, nil, "سجح", nil, nil, nil],
      64862 => [0, 0, nil, "سجى", nil, nil, nil],
      64863 => [0, 0, nil, "سمح", nil, nil, nil],
      64864 => [0, 0, nil, "سمح", nil, nil, nil],
      64865 => [0, 0, nil, "سمج", nil, nil, nil],
      64866 => [0, 0, nil, "سمم", nil, nil, nil],
      64867 => [0, 0, nil, "سمم", nil, nil, nil],
      64868 => [0, 0, nil, "صحح", nil, nil, nil],
      64869 => [0, 0, nil, "صحح", nil, nil, nil],
      64870 => [0, 0, nil, "صمم", nil, nil, nil],
      64871 => [0, 0, nil, "شحم", nil, nil, nil],
      64872 => [0, 0, nil, "شحم", nil, nil, nil],
      64873 => [0, 0, nil, "شجي", nil, nil, nil],
      64874 => [0, 0, nil, "شمخ", nil, nil, nil],
      64875 => [0, 0, nil, "شمخ", nil, nil, nil],
      64876 => [0, 0, nil, "شمم", nil, nil, nil],
      64877 => [0, 0, nil, "شمم", nil, nil, nil],
      64878 => [0, 0, nil, "ضحى", nil, nil, nil],
      64879 => [0, 0, nil, "ضخم", nil, nil, nil],
      64880 => [0, 0, nil, "ضخم", nil, nil, nil],
      64881 => [0, 0, nil, "طمح", nil, nil, nil],
      64882 => [0, 0, nil, "طمح", nil, nil, nil],
      64883 => [0, 0, nil, "طمم", nil, nil, nil],
      64884 => [0, 0, nil, "طمي", nil, nil, nil],
      64885 => [0, 0, nil, "عجم", nil, nil, nil],
      64886 => [0, 0, nil, "عمم", nil, nil, nil],
      64887 => [0, 0, nil, "عمم", nil, nil, nil],
      64888 => [0, 0, nil, "عمى", nil, nil, nil],
      64889 => [0, 0, nil, "غمم", nil, nil, nil],
      64890 => [0, 0, nil, "غمي", nil, nil, nil],
      64891 => [0, 0, nil, "غمى", nil, nil, nil],
      64892 => [0, 0, nil, "فخم", nil, nil, nil],
      64893 => [0, 0, nil, "فخم", nil, nil, nil],
      64894 => [0, 0, nil, "قمح", nil, nil, nil],
      64895 => [0, 0, nil, "قمم", nil, nil, nil],
      64896 => [0, 0, nil, "لحم", nil, nil, nil],
      64897 => [0, 0, nil, "لحي", nil, nil, nil],
      64898 => [0, 0, nil, "لحى", nil, nil, nil],
      64899 => [0, 0, nil, "لجج", nil, nil, nil],
      64900 => [0, 0, nil, "لجج", nil, nil, nil],
      64901 => [0, 0, nil, "لخم", nil, nil, nil],
      64902 => [0, 0, nil, "لخم", nil, nil, nil],
      64903 => [0, 0, nil, "لمح", nil, nil, nil],
      64904 => [0, 0, nil, "لمح", nil, nil, nil],
      64905 => [0, 0, nil, "محج", nil, nil, nil],
      64906 => [0, 0, nil, "محم", nil, nil, nil],
      64907 => [0, 0, nil, "محي", nil, nil, nil],
      64908 => [0, 0, nil, "مجح", nil, nil, nil],
      64909 => [0, 0, nil, "مجم", nil, nil, nil],
      64910 => [0, 0, nil, "مخج", nil, nil, nil],
      64911 => [0, 0, nil, "مخم", nil, nil, nil],
      64914 => [0, 0, nil, "مجخ", nil, nil, nil],
      64915 => [0, 0, nil, "همج", nil, nil, nil],
      64916 => [0, 0, nil, "همم", nil, nil, nil],
      64917 => [0, 0, nil, "نحم", nil, nil, nil],
      64918 => [0, 0, nil, "نحى", nil, nil, nil],
      64919 => [0, 0, nil, "نجم", nil, nil, nil],
      64920 => [0, 0, nil, "نجم", nil, nil, nil],
      64921 => [0, 0, nil, "نجى", nil, nil, nil],
      64922 => [0, 0, nil, "نمي", nil, nil, nil],
      64923 => [0, 0, nil, "نمى", nil, nil, nil],
      64924 => [0, 0, nil, "يمم", nil, nil, nil],
      64925 => [0, 0, nil, "يمم", nil, nil, nil],
      64926 => [0, 0, nil, "بخي", nil, nil, nil],
      64927 => [0, 0, nil, "تجي", nil, nil, nil],
      64928 => [0, 0, nil, "تجى", nil, nil, nil],
      64929 => [0, 0, nil, "تخي", nil, nil, nil],
      64930 => [0, 0, nil, "تخى", nil, nil, nil],
      64931 => [0, 0, nil, "تمي", nil, nil, nil],
      64932 => [0, 0, nil, "تمى", nil, nil, nil],
      64933 => [0, 0, nil, "جمي", nil, nil, nil],
      64934 => [0, 0, nil, "جحى", nil, nil, nil],
      64935 => [0, 0, nil, "جمى", nil, nil, nil],
      64936 => [0, 0, nil, "سخى", nil, nil, nil],
      64937 => [0, 0, nil, "صحي", nil, nil, nil],
      64938 => [0, 0, nil, "شحي", nil, nil, nil],
      64939 => [0, 0, nil, "ضحي", nil, nil, nil],
      64940 => [0, 0, nil, "لجي", nil, nil, nil],
      64941 => [0, 0, nil, "لمي", nil, nil, nil],
      64942 => [0, 0, nil, "يحي", nil, nil, nil],
      64943 => [0, 0, nil, "يجي", nil, nil, nil],
      64944 => [0, 0, nil, "يمي", nil, nil, nil],
      64945 => [0, 0, nil, "ممي", nil, nil, nil],
      64946 => [0, 0, nil, "قمي", nil, nil, nil],
      64947 => [0, 0, nil, "نحي", nil, nil, nil],
      64948 => [0, 0, nil, "قمح", nil, nil, nil],
      64949 => [0, 0, nil, "لحم", nil, nil, nil],
      64950 => [0, 0, nil, "عمي", nil, nil, nil],
      64951 => [0, 0, nil, "كمي", nil, nil, nil],
      64952 => [0, 0, nil, "نجح", nil, nil, nil],
      64953 => [0, 0, nil, "مخي", nil, nil, nil],
      64954 => [0, 0, nil, "لجم", nil, nil, nil],
      64955 => [0, 0, nil, "كمم", nil, nil, nil],
      64956 => [0, 0, nil, "لجم", nil, nil, nil],
      64957 => [0, 0, nil, "نجح", nil, nil, nil],
      64958 => [0, 0, nil, "جحي", nil, nil, nil],
      64959 => [0, 0, nil, "حجي", nil, nil, nil],
      64960 => [0, 0, nil, "مجي", nil, nil, nil],
      64961 => [0, 0, nil, "فمي", nil, nil, nil],
      64962 => [0, 0, nil, "بحي", nil, nil, nil],
      64963 => [0, 0, nil, "كمم", nil, nil, nil],
      64964 => [0, 0, nil, "عجم", nil, nil, nil],
      64965 => [0, 0, nil, "صمم", nil, nil, nil],
      64966 => [0, 0, nil, "سخي", nil, nil, nil],
      64967 => [0, 0, nil, "نجي", nil, nil, nil],
      65008 => [0, 0, nil, "صلے", nil, nil, nil],
      65009 => [0, 0, nil, "قلے", nil, nil, nil],
      65010 => [0, 0, nil, "الله", nil, nil, nil],
      65011 => [0, 0, nil, "اكبر", nil, nil, nil],
      65012 => [0, 0, nil, "محمد", nil, nil, nil],
      65013 => [0, 0, nil, "صلعم", nil, nil, nil],
      65014 => [0, 0, nil, "رسول", nil, nil, nil],
      65015 => [0, 0, nil, "عليه", nil, nil, nil],
      65016 => [0, 0, nil, "وسلم", nil, nil, nil],
      65017 => [0, 0, nil, "صلى", nil, nil, nil],
      65018 => [0, 0, nil, "صلى الله عليه وسلم", nil, nil, nil],
      65019 => [0, 0, nil, "جل جلاله", nil, nil, nil],
      65056 => [230, 0, nil, nil, nil, nil, nil],
      65057 => [230, 0, nil, nil, nil, nil, nil],
      65058 => [230, 0, nil, nil, nil, nil, nil],
      65059 => [230, 0, nil, nil, nil, nil, nil],
      65072 => [0, 0, nil, "‥", nil, nil, nil],
      65073 => [0, 0, nil, "—", nil, nil, nil],
      65074 => [0, 0, nil, "–", nil, nil, nil],
      65075 => [0, 0, nil, "_", nil, nil, nil],
      65076 => [0, 0, nil, "_", nil, nil, nil],
      65077 => [0, 0, nil, "(", nil, nil, nil],
      65078 => [0, 0, nil, ")", nil, nil, nil],
      65079 => [0, 0, nil, "{", nil, nil, nil],
      65080 => [0, 0, nil, "}", nil, nil, nil],
      65081 => [0, 0, nil, "〔", nil, nil, nil],
      65082 => [0, 0, nil, "〕", nil, nil, nil],
      65083 => [0, 0, nil, "【", nil, nil, nil],
      65084 => [0, 0, nil, "】", nil, nil, nil],
      65085 => [0, 0, nil, "《", nil, nil, nil],
      65086 => [0, 0, nil, "》", nil, nil, nil],
      65087 => [0, 0, nil, "〈", nil, nil, nil],
      65088 => [0, 0, nil, "〉", nil, nil, nil],
      65089 => [0, 0, nil, "「", nil, nil, nil],
      65090 => [0, 0, nil, "」", nil, nil, nil],
      65091 => [0, 0, nil, "『", nil, nil, nil],
      65092 => [0, 0, nil, "』", nil, nil, nil],
      65097 => [0, 0, nil, "‾", nil, nil, nil],
      65098 => [0, 0, nil, "‾", nil, nil, nil],
      65099 => [0, 0, nil, "‾", nil, nil, nil],
      65100 => [0, 0, nil, "‾", nil, nil, nil],
      65101 => [0, 0, nil, "_", nil, nil, nil],
      65102 => [0, 0, nil, "_", nil, nil, nil],
      65103 => [0, 0, nil, "_", nil, nil, nil],
      65104 => [0, 0, nil, ",", nil, nil, nil],
      65105 => [0, 0, nil, "、", nil, nil, nil],
      65106 => [0, 0, nil, ".", nil, nil, nil],
      65108 => [0, 0, nil, ";", nil, nil, nil],
      65109 => [0, 0, nil, ":", nil, nil, nil],
      65110 => [0, 0, nil, "?", nil, nil, nil],
      65111 => [0, 0, nil, "!", nil, nil, nil],
      65112 => [0, 0, nil, "—", nil, nil, nil],
      65113 => [0, 0, nil, "(", nil, nil, nil],
      65114 => [0, 0, nil, ")", nil, nil, nil],
      65115 => [0, 0, nil, "{", nil, nil, nil],
      65116 => [0, 0, nil, "}", nil, nil, nil],
      65117 => [0, 0, nil, "〔", nil, nil, nil],
      65118 => [0, 0, nil, "〕", nil, nil, nil],
      65119 => [0, 0, nil, "#", nil, nil, nil],
      65120 => [0, 0, nil, "&", nil, nil, nil],
      65121 => [0, 0, nil, "*", nil, nil, nil],
      65122 => [0, 0, nil, "+", nil, nil, nil],
      65123 => [0, 0, nil, "-", nil, nil, nil],
      65124 => [0, 0, nil, "<", nil, nil, nil],
      65125 => [0, 0, nil, ">", nil, nil, nil],
      65126 => [0, 0, nil, "=", nil, nil, nil],
      65128 => [0, 0, nil, "\\", nil, nil, nil],
      65129 => [0, 0, nil, "$", nil, nil, nil],
      65130 => [0, 0, nil, "%", nil, nil, nil],
      65131 => [0, 0, nil, "@", nil, nil, nil],
      65136 => [0, 0, nil, " ً", nil, nil, nil],
      65137 => [0, 0, nil, "ـً", nil, nil, nil],
      65138 => [0, 0, nil, " ٌ", nil, nil, nil],
      65140 => [0, 0, nil, " ٍ", nil, nil, nil],
      65142 => [0, 0, nil, " َ", nil, nil, nil],
      65143 => [0, 0, nil, "ـَ", nil, nil, nil],
      65144 => [0, 0, nil, " ُ", nil, nil, nil],
      65145 => [0, 0, nil, "ـُ", nil, nil, nil],
      65146 => [0, 0, nil, " ِ", nil, nil, nil],
      65147 => [0, 0, nil, "ـِ", nil, nil, nil],
      65148 => [0, 0, nil, " ّ", nil, nil, nil],
      65149 => [0, 0, nil, "ـّ", nil, nil, nil],
      65150 => [0, 0, nil, " ْ", nil, nil, nil],
      65151 => [0, 0, nil, "ـْ", nil, nil, nil],
      65152 => [0, 0, nil, "ء", nil, nil, nil],
      65153 => [0, 0, nil, "آ", nil, nil, nil],
      65154 => [0, 0, nil, "آ", nil, nil, nil],
      65155 => [0, 0, nil, "أ", nil, nil, nil],
      65156 => [0, 0, nil, "أ", nil, nil, nil],
      65157 => [0, 0, nil, "ؤ", nil, nil, nil],
      65158 => [0, 0, nil, "ؤ", nil, nil, nil],
      65159 => [0, 0, nil, "إ", nil, nil, nil],
      65160 => [0, 0, nil, "إ", nil, nil, nil],
      65161 => [0, 0, nil, "ئ", nil, nil, nil],
      65162 => [0, 0, nil, "ئ", nil, nil, nil],
      65163 => [0, 0, nil, "ئ", nil, nil, nil],
      65164 => [0, 0, nil, "ئ", nil, nil, nil],
      65165 => [0, 0, nil, "ا", nil, nil, nil],
      65166 => [0, 0, nil, "ا", nil, nil, nil],
      65167 => [0, 0, nil, "ب", nil, nil, nil],
      65168 => [0, 0, nil, "ب", nil, nil, nil],
      65169 => [0, 0, nil, "ب", nil, nil, nil],
      65170 => [0, 0, nil, "ب", nil, nil, nil],
      65171 => [0, 0, nil, "ة", nil, nil, nil],
      65172 => [0, 0, nil, "ة", nil, nil, nil],
      65173 => [0, 0, nil, "ت", nil, nil, nil],
      65174 => [0, 0, nil, "ت", nil, nil, nil],
      65175 => [0, 0, nil, "ت", nil, nil, nil],
      65176 => [0, 0, nil, "ت", nil, nil, nil],
      65177 => [0, 0, nil, "ث", nil, nil, nil],
      65178 => [0, 0, nil, "ث", nil, nil, nil],
      65179 => [0, 0, nil, "ث", nil, nil, nil],
      65180 => [0, 0, nil, "ث", nil, nil, nil],
      65181 => [0, 0, nil, "ج", nil, nil, nil],
      65182 => [0, 0, nil, "ج", nil, nil, nil],
      65183 => [0, 0, nil, "ج", nil, nil, nil],
      65184 => [0, 0, nil, "ج", nil, nil, nil],
      65185 => [0, 0, nil, "ح", nil, nil, nil],
      65186 => [0, 0, nil, "ح", nil, nil, nil],
      65187 => [0, 0, nil, "ح", nil, nil, nil],
      65188 => [0, 0, nil, "ح", nil, nil, nil],
      65189 => [0, 0, nil, "خ", nil, nil, nil],
      65190 => [0, 0, nil, "خ", nil, nil, nil],
      65191 => [0, 0, nil, "خ", nil, nil, nil],
      65192 => [0, 0, nil, "خ", nil, nil, nil],
      65193 => [0, 0, nil, "د", nil, nil, nil],
      65194 => [0, 0, nil, "د", nil, nil, nil],
      65195 => [0, 0, nil, "ذ", nil, nil, nil],
      65196 => [0, 0, nil, "ذ", nil, nil, nil],
      65197 => [0, 0, nil, "ر", nil, nil, nil],
      65198 => [0, 0, nil, "ر", nil, nil, nil],
      65199 => [0, 0, nil, "ز", nil, nil, nil],
      65200 => [0, 0, nil, "ز", nil, nil, nil],
      65201 => [0, 0, nil, "س", nil, nil, nil],
      65202 => [0, 0, nil, "س", nil, nil, nil],
      65203 => [0, 0, nil, "س", nil, nil, nil],
      65204 => [0, 0, nil, "س", nil, nil, nil],
      65205 => [0, 0, nil, "ش", nil, nil, nil],
      65206 => [0, 0, nil, "ش", nil, nil, nil],
      65207 => [0, 0, nil, "ش", nil, nil, nil],
      65208 => [0, 0, nil, "ش", nil, nil, nil],
      65209 => [0, 0, nil, "ص", nil, nil, nil],
      65210 => [0, 0, nil, "ص", nil, nil, nil],
      65211 => [0, 0, nil, "ص", nil, nil, nil],
      65212 => [0, 0, nil, "ص", nil, nil, nil],
      65213 => [0, 0, nil, "ض", nil, nil, nil],
      65214 => [0, 0, nil, "ض", nil, nil, nil],
      65215 => [0, 0, nil, "ض", nil, nil, nil],
      65216 => [0, 0, nil, "ض", nil, nil, nil],
      65217 => [0, 0, nil, "ط", nil, nil, nil],
      65218 => [0, 0, nil, "ط", nil, nil, nil],
      65219 => [0, 0, nil, "ط", nil, nil, nil],
      65220 => [0, 0, nil, "ط", nil, nil, nil],
      65221 => [0, 0, nil, "ظ", nil, nil, nil],
      65222 => [0, 0, nil, "ظ", nil, nil, nil],
      65223 => [0, 0, nil, "ظ", nil, nil, nil],
      65224 => [0, 0, nil, "ظ", nil, nil, nil],
      65225 => [0, 0, nil, "ع", nil, nil, nil],
      65226 => [0, 0, nil, "ع", nil, nil, nil],
      65227 => [0, 0, nil, "ع", nil, nil, nil],
      65228 => [0, 0, nil, "ع", nil, nil, nil],
      65229 => [0, 0, nil, "غ", nil, nil, nil],
      65230 => [0, 0, nil, "غ", nil, nil, nil],
      65231 => [0, 0, nil, "غ", nil, nil, nil],
      65232 => [0, 0, nil, "غ", nil, nil, nil],
      65233 => [0, 0, nil, "ف", nil, nil, nil],
      65234 => [0, 0, nil, "ف", nil, nil, nil],
      65235 => [0, 0, nil, "ف", nil, nil, nil],
      65236 => [0, 0, nil, "ف", nil, nil, nil],
      65237 => [0, 0, nil, "ق", nil, nil, nil],
      65238 => [0, 0, nil, "ق", nil, nil, nil],
      65239 => [0, 0, nil, "ق", nil, nil, nil],
      65240 => [0, 0, nil, "ق", nil, nil, nil],
      65241 => [0, 0, nil, "ك", nil, nil, nil],
      65242 => [0, 0, nil, "ك", nil, nil, nil],
      65243 => [0, 0, nil, "ك", nil, nil, nil],
      65244 => [0, 0, nil, "ك", nil, nil, nil],
      65245 => [0, 0, nil, "ل", nil, nil, nil],
      65246 => [0, 0, nil, "ل", nil, nil, nil],
      65247 => [0, 0, nil, "ل", nil, nil, nil],
      65248 => [0, 0, nil, "ل", nil, nil, nil],
      65249 => [0, 0, nil, "م", nil, nil, nil],
      65250 => [0, 0, nil, "م", nil, nil, nil],
      65251 => [0, 0, nil, "م", nil, nil, nil],
      65252 => [0, 0, nil, "م", nil, nil, nil],
      65253 => [0, 0, nil, "ن", nil, nil, nil],
      65254 => [0, 0, nil, "ن", nil, nil, nil],
      65255 => [0, 0, nil, "ن", nil, nil, nil],
      65256 => [0, 0, nil, "ن", nil, nil, nil],
      65257 => [0, 0, nil, "ه", nil, nil, nil],
      65258 => [0, 0, nil, "ه", nil, nil, nil],
      65259 => [0, 0, nil, "ه", nil, nil, nil],
      65260 => [0, 0, nil, "ه", nil, nil, nil],
      65261 => [0, 0, nil, "و", nil, nil, nil],
      65262 => [0, 0, nil, "و", nil, nil, nil],
      65263 => [0, 0, nil, "ى", nil, nil, nil],
      65264 => [0, 0, nil, "ى", nil, nil, nil],
      65265 => [0, 0, nil, "ي", nil, nil, nil],
      65266 => [0, 0, nil, "ي", nil, nil, nil],
      65267 => [0, 0, nil, "ي", nil, nil, nil],
      65268 => [0, 0, nil, "ي", nil, nil, nil],
      65269 => [0, 0, nil, "لآ", nil, nil, nil],
      65270 => [0, 0, nil, "لآ", nil, nil, nil],
      65271 => [0, 0, nil, "لأ", nil, nil, nil],
      65272 => [0, 0, nil, "لأ", nil, nil, nil],
      65273 => [0, 0, nil, "لإ", nil, nil, nil],
      65274 => [0, 0, nil, "لإ", nil, nil, nil],
      65275 => [0, 0, nil, "لا", nil, nil, nil],
      65276 => [0, 0, nil, "لا", nil, nil, nil],
      65281 => [0, 0, nil, "!", nil, nil, nil],
      65282 => [0, 0, nil, "\"", nil, nil, nil],
      65283 => [0, 0, nil, "#", nil, nil, nil],
      65284 => [0, 0, nil, "$", nil, nil, nil],
      65285 => [0, 0, nil, "%", nil, nil, nil],
      65286 => [0, 0, nil, "&", nil, nil, nil],
      65287 => [0, 0, nil, "'", nil, nil, nil],
      65288 => [0, 0, nil, "(", nil, nil, nil],
      65289 => [0, 0, nil, ")", nil, nil, nil],
      65290 => [0, 0, nil, "*", nil, nil, nil],
      65291 => [0, 0, nil, "+", nil, nil, nil],
      65292 => [0, 0, nil, ",", nil, nil, nil],
      65293 => [0, 0, nil, "-", nil, nil, nil],
      65294 => [0, 0, nil, ".", nil, nil, nil],
      65295 => [0, 0, nil, "/", nil, nil, nil],
      65296 => [0, 0, nil, "0", nil, nil, nil],
      65297 => [0, 0, nil, "1", nil, nil, nil],
      65298 => [0, 0, nil, "2", nil, nil, nil],
      65299 => [0, 0, nil, "3", nil, nil, nil],
      65300 => [0, 0, nil, "4", nil, nil, nil],
      65301 => [0, 0, nil, "5", nil, nil, nil],
      65302 => [0, 0, nil, "6", nil, nil, nil],
      65303 => [0, 0, nil, "7", nil, nil, nil],
      65304 => [0, 0, nil, "8", nil, nil, nil],
      65305 => [0, 0, nil, "9", nil, nil, nil],
      65306 => [0, 0, nil, ":", nil, nil, nil],
      65307 => [0, 0, nil, ";", nil, nil, nil],
      65308 => [0, 0, nil, "<", nil, nil, nil],
      65309 => [0, 0, nil, "=", nil, nil, nil],
      65310 => [0, 0, nil, ">", nil, nil, nil],
      65311 => [0, 0, nil, "?", nil, nil, nil],
      65312 => [0, 0, nil, "@", nil, nil, nil],
      65313 => [0, 0, nil, "A", nil, 65345, nil],
      65314 => [0, 0, nil, "B", nil, 65346, nil],
      65315 => [0, 0, nil, "C", nil, 65347, nil],
      65316 => [0, 0, nil, "D", nil, 65348, nil],
      65317 => [0, 0, nil, "E", nil, 65349, nil],
      65318 => [0, 0, nil, "F", nil, 65350, nil],
      65319 => [0, 0, nil, "G", nil, 65351, nil],
      65320 => [0, 0, nil, "H", nil, 65352, nil],
      65321 => [0, 0, nil, "I", nil, 65353, nil],
      65322 => [0, 0, nil, "J", nil, 65354, nil],
      65323 => [0, 0, nil, "K", nil, 65355, nil],
      65324 => [0, 0, nil, "L", nil, 65356, nil],
      65325 => [0, 0, nil, "M", nil, 65357, nil],
      65326 => [0, 0, nil, "N", nil, 65358, nil],
      65327 => [0, 0, nil, "O", nil, 65359, nil],
      65328 => [0, 0, nil, "P", nil, 65360, nil],
      65329 => [0, 0, nil, "Q", nil, 65361, nil],
      65330 => [0, 0, nil, "R", nil, 65362, nil],
      65331 => [0, 0, nil, "S", nil, 65363, nil],
      65332 => [0, 0, nil, "T", nil, 65364, nil],
      65333 => [0, 0, nil, "U", nil, 65365, nil],
      65334 => [0, 0, nil, "V", nil, 65366, nil],
      65335 => [0, 0, nil, "W", nil, 65367, nil],
      65336 => [0, 0, nil, "X", nil, 65368, nil],
      65337 => [0, 0, nil, "Y", nil, 65369, nil],
      65338 => [0, 0, nil, "Z", nil, 65370, nil],
      65339 => [0, 0, nil, "[", nil, nil, nil],
      65340 => [0, 0, nil, "\\", nil, nil, nil],
      65341 => [0, 0, nil, "]", nil, nil, nil],
      65342 => [0, 0, nil, "^", nil, nil, nil],
      65343 => [0, 0, nil, "_", nil, nil, nil],
      65344 => [0, 0, nil, "`", nil, nil, nil],
      65345 => [0, 0, nil, "a", 65313, nil, 65313],
      65346 => [0, 0, nil, "b", 65314, nil, 65314],
      65347 => [0, 0, nil, "c", 65315, nil, 65315],
      65348 => [0, 0, nil, "d", 65316, nil, 65316],
      65349 => [0, 0, nil, "e", 65317, nil, 65317],
      65350 => [0, 0, nil, "f", 65318, nil, 65318],
      65351 => [0, 0, nil, "g", 65319, nil, 65319],
      65352 => [0, 0, nil, "h", 65320, nil, 65320],
      65353 => [0, 0, nil, "i", 65321, nil, 65321],
      65354 => [0, 0, nil, "j", 65322, nil, 65322],
      65355 => [0, 0, nil, "k", 65323, nil, 65323],
      65356 => [0, 0, nil, "l", 65324, nil, 65324],
      65357 => [0, 0, nil, "m", 65325, nil, 65325],
      65358 => [0, 0, nil, "n", 65326, nil, 65326],
      65359 => [0, 0, nil, "o", 65327, nil, 65327],
      65360 => [0, 0, nil, "p", 65328, nil, 65328],
      65361 => [0, 0, nil, "q", 65329, nil, 65329],
      65362 => [0, 0, nil, "r", 65330, nil, 65330],
      65363 => [0, 0, nil, "s", 65331, nil, 65331],
      65364 => [0, 0, nil, "t", 65332, nil, 65332],
      65365 => [0, 0, nil, "u", 65333, nil, 65333],
      65366 => [0, 0, nil, "v", 65334, nil, 65334],
      65367 => [0, 0, nil, "w", 65335, nil, 65335],
      65368 => [0, 0, nil, "x", 65336, nil, 65336],
      65369 => [0, 0, nil, "y", 65337, nil, 65337],
      65370 => [0, 0, nil, "z", 65338, nil, 65338],
      65371 => [0, 0, nil, "{", nil, nil, nil],
      65372 => [0, 0, nil, "|", nil, nil, nil],
      65373 => [0, 0, nil, "}", nil, nil, nil],
      65374 => [0, 0, nil, "~", nil, nil, nil],
      65377 => [0, 0, nil, "。", nil, nil, nil],
      65378 => [0, 0, nil, "「", nil, nil, nil],
      65379 => [0, 0, nil, "」", nil, nil, nil],
      65380 => [0, 0, nil, "、", nil, nil, nil],
      65381 => [0, 0, nil, "・", nil, nil, nil],
      65382 => [0, 0, nil, "ヲ", nil, nil, nil],
      65383 => [0, 0, nil, "ァ", nil, nil, nil],
      65384 => [0, 0, nil, "ィ", nil, nil, nil],
      65385 => [0, 0, nil, "ゥ", nil, nil, nil],
      65386 => [0, 0, nil, "ェ", nil, nil, nil],
      65387 => [0, 0, nil, "ォ", nil, nil, nil],
      65388 => [0, 0, nil, "ャ", nil, nil, nil],
      65389 => [0, 0, nil, "ュ", nil, nil, nil],
      65390 => [0, 0, nil, "ョ", nil, nil, nil],
      65391 => [0, 0, nil, "ッ", nil, nil, nil],
      65392 => [0, 0, nil, "ー", nil, nil, nil],
      65393 => [0, 0, nil, "ア", nil, nil, nil],
      65394 => [0, 0, nil, "イ", nil, nil, nil],
      65395 => [0, 0, nil, "ウ", nil, nil, nil],
      65396 => [0, 0, nil, "エ", nil, nil, nil],
      65397 => [0, 0, nil, "オ", nil, nil, nil],
      65398 => [0, 0, nil, "カ", nil, nil, nil],
      65399 => [0, 0, nil, "キ", nil, nil, nil],
      65400 => [0, 0, nil, "ク", nil, nil, nil],
      65401 => [0, 0, nil, "ケ", nil, nil, nil],
      65402 => [0, 0, nil, "コ", nil, nil, nil],
      65403 => [0, 0, nil, "サ", nil, nil, nil],
      65404 => [0, 0, nil, "シ", nil, nil, nil],
      65405 => [0, 0, nil, "ス", nil, nil, nil],
      65406 => [0, 0, nil, "セ", nil, nil, nil],
      65407 => [0, 0, nil, "ソ", nil, nil, nil],
      65408 => [0, 0, nil, "タ", nil, nil, nil],
      65409 => [0, 0, nil, "チ", nil, nil, nil],
      65410 => [0, 0, nil, "ツ", nil, nil, nil],
      65411 => [0, 0, nil, "テ", nil, nil, nil],
      65412 => [0, 0, nil, "ト", nil, nil, nil],
      65413 => [0, 0, nil, "ナ", nil, nil, nil],
      65414 => [0, 0, nil, "ニ", nil, nil, nil],
      65415 => [0, 0, nil, "ヌ", nil, nil, nil],
      65416 => [0, 0, nil, "ネ", nil, nil, nil],
      65417 => [0, 0, nil, "ノ", nil, nil, nil],
      65418 => [0, 0, nil, "ハ", nil, nil, nil],
      65419 => [0, 0, nil, "ヒ", nil, nil, nil],
      65420 => [0, 0, nil, "フ", nil, nil, nil],
      65421 => [0, 0, nil, "ヘ", nil, nil, nil],
      65422 => [0, 0, nil, "ホ", nil, nil, nil],
      65423 => [0, 0, nil, "マ", nil, nil, nil],
      65424 => [0, 0, nil, "ミ", nil, nil, nil],
      65425 => [0, 0, nil, "ム", nil, nil, nil],
      65426 => [0, 0, nil, "メ", nil, nil, nil],
      65427 => [0, 0, nil, "モ", nil, nil, nil],
      65428 => [0, 0, nil, "ヤ", nil, nil, nil],
      65429 => [0, 0, nil, "ユ", nil, nil, nil],
      65430 => [0, 0, nil, "ヨ", nil, nil, nil],
      65431 => [0, 0, nil, "ラ", nil, nil, nil],
      65432 => [0, 0, nil, "リ", nil, nil, nil],
      65433 => [0, 0, nil, "ル", nil, nil, nil],
      65434 => [0, 0, nil, "レ", nil, nil, nil],
      65435 => [0, 0, nil, "ロ", nil, nil, nil],
      65436 => [0, 0, nil, "ワ", nil, nil, nil],
      65437 => [0, 0, nil, "ン", nil, nil, nil],
      65438 => [0, 0, nil, "゙", nil, nil, nil],
      65439 => [0, 0, nil, "゚", nil, nil, nil],
      65440 => [0, 0, nil, "ㅤ", nil, nil, nil],
      65441 => [0, 0, nil, "ㄱ", nil, nil, nil],
      65442 => [0, 0, nil, "ㄲ", nil, nil, nil],
      65443 => [0, 0, nil, "ㄳ", nil, nil, nil],
      65444 => [0, 0, nil, "ㄴ", nil, nil, nil],
      65445 => [0, 0, nil, "ㄵ", nil, nil, nil],
      65446 => [0, 0, nil, "ㄶ", nil, nil, nil],
      65447 => [0, 0, nil, "ㄷ", nil, nil, nil],
      65448 => [0, 0, nil, "ㄸ", nil, nil, nil],
      65449 => [0, 0, nil, "ㄹ", nil, nil, nil],
      65450 => [0, 0, nil, "ㄺ", nil, nil, nil],
      65451 => [0, 0, nil, "ㄻ", nil, nil, nil],
      65452 => [0, 0, nil, "ㄼ", nil, nil, nil],
      65453 => [0, 0, nil, "ㄽ", nil, nil, nil],
      65454 => [0, 0, nil, "ㄾ", nil, nil, nil],
      65455 => [0, 0, nil, "ㄿ", nil, nil, nil],
      65456 => [0, 0, nil, "ㅀ", nil, nil, nil],
      65457 => [0, 0, nil, "ㅁ", nil, nil, nil],
      65458 => [0, 0, nil, "ㅂ", nil, nil, nil],
      65459 => [0, 0, nil, "ㅃ", nil, nil, nil],
      65460 => [0, 0, nil, "ㅄ", nil, nil, nil],
      65461 => [0, 0, nil, "ㅅ", nil, nil, nil],
      65462 => [0, 0, nil, "ㅆ", nil, nil, nil],
      65463 => [0, 0, nil, "ㅇ", nil, nil, nil],
      65464 => [0, 0, nil, "ㅈ", nil, nil, nil],
      65465 => [0, 0, nil, "ㅉ", nil, nil, nil],
      65466 => [0, 0, nil, "ㅊ", nil, nil, nil],
      65467 => [0, 0, nil, "ㅋ", nil, nil, nil],
      65468 => [0, 0, nil, "ㅌ", nil, nil, nil],
      65469 => [0, 0, nil, "ㅍ", nil, nil, nil],
      65470 => [0, 0, nil, "ㅎ", nil, nil, nil],
      65474 => [0, 0, nil, "ㅏ", nil, nil, nil],
      65475 => [0, 0, nil, "ㅐ", nil, nil, nil],
      65476 => [0, 0, nil, "ㅑ", nil, nil, nil],
      65477 => [0, 0, nil, "ㅒ", nil, nil, nil],
      65478 => [0, 0, nil, "ㅓ", nil, nil, nil],
      65479 => [0, 0, nil, "ㅔ", nil, nil, nil],
      65482 => [0, 0, nil, "ㅕ", nil, nil, nil],
      65483 => [0, 0, nil, "ㅖ", nil, nil, nil],
      65484 => [0, 0, nil, "ㅗ", nil, nil, nil],
      65485 => [0, 0, nil, "ㅘ", nil, nil, nil],
      65486 => [0, 0, nil, "ㅙ", nil, nil, nil],
      65487 => [0, 0, nil, "ㅚ", nil, nil, nil],
      65490 => [0, 0, nil, "ㅛ", nil, nil, nil],
      65491 => [0, 0, nil, "ㅜ", nil, nil, nil],
      65492 => [0, 0, nil, "ㅝ", nil, nil, nil],
      65493 => [0, 0, nil, "ㅞ", nil, nil, nil],
      65494 => [0, 0, nil, "ㅟ", nil, nil, nil],
      65495 => [0, 0, nil, "ㅠ", nil, nil, nil],
      65498 => [0, 0, nil, "ㅡ", nil, nil, nil],
      65499 => [0, 0, nil, "ㅢ", nil, nil, nil],
      65500 => [0, 0, nil, "ㅣ", nil, nil, nil],
      65504 => [0, 0, nil, "¢", nil, nil, nil],
      65505 => [0, 0, nil, "£", nil, nil, nil],
      65506 => [0, 0, nil, "¬", nil, nil, nil],
      65507 => [0, 0, nil, "¯", nil, nil, nil],
      65508 => [0, 0, nil, "¦", nil, nil, nil],
      65509 => [0, 0, nil, "¥", nil, nil, nil],
      65510 => [0, 0, nil, "₩", nil, nil, nil],
      65512 => [0, 0, nil, "│", nil, nil, nil],
      65513 => [0, 0, nil, "←", nil, nil, nil],
      65514 => [0, 0, nil, "↑", nil, nil, nil],
      65515 => [0, 0, nil, "→", nil, nil, nil],
      65516 => [0, 0, nil, "↓", nil, nil, nil],
      65517 => [0, 0, nil, "■", nil, nil, nil],
      65518 => [0, 0, nil, "○", nil, nil, nil],
    }

    COMPOSITION_TABLE = {}
    UNICODE_DATA.each do |codepoint, data|
      canonical = data[UNICODE_DATA_CANONICAL]
      exclusion = data[UNICODE_DATA_EXCLUSION]

      if canonical && exclusion == 0
        COMPOSITION_TABLE[canonical.unpack("C*")] = codepoint
      end
    end

    UNICODE_MAX_LENGTH = 256
    ACE_MAX_LENGTH = 256

    PUNYCODE_BASE = 36
    PUNYCODE_TMIN = 1
    PUNYCODE_TMAX = 26
    PUNYCODE_SKEW = 38
    PUNYCODE_DAMP = 700
    PUNYCODE_INITIAL_BIAS = 72
    PUNYCODE_INITIAL_N = 0x80
    PUNYCODE_DELIMITER = 0x2D

    PUNYCODE_MAXINT = 1 << 64

    PUNYCODE_PRINT_ASCII =
      "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n" +
      "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n" +
      " !\"\#$%&'()*+,-./" +
      "0123456789:;<=>?" +
      "@ABCDEFGHIJKLMNO" +
      "PQRSTUVWXYZ[\\]^_" +
      "`abcdefghijklmno" +
      "pqrstuvwxyz{|}~\n"

    # Input is invalid.
    class PunycodeBadInput < StandardError; end
    # Output would exceed the space provided.
    class PunycodeBigOutput < StandardError; end
    # Input needs wider integers to process.
    class PunycodeOverflow < StandardError; end

    def self.punycode_encode(unicode)
      unicode = unicode.to_s unless unicode.is_a?(String)
      input = unicode.unpack("U*")
      output = [0] * (ACE_MAX_LENGTH + 1)
      input_length = input.size
      output_length = [ACE_MAX_LENGTH]

      # Initialize the state
      n = PUNYCODE_INITIAL_N
      delta = out = 0
      max_out = output_length[0]
      bias = PUNYCODE_INITIAL_BIAS

      # Handle the basic code points:
      input_length.times do |j|
        if punycode_basic?(input[j])
          if max_out - out < 2
            raise PunycodeBigOutput,
              "Output would exceed the space provided."
          end
          output[out] = input[j]
          out += 1
        end
      end

      h = b = out

      # h is the number of code points that have been handled, b is the
      # number of basic code points, and out is the number of characters
      # that have been output.

      if b > 0
        output[out] = PUNYCODE_DELIMITER
        out += 1
      end

      # Main encoding loop:

      while h < input_length
        # All non-basic code points < n have been
        # handled already.  Find the next larger one:

        m = PUNYCODE_MAXINT
        input_length.times do |j|
          m = input[j] if (n...m) === input[j]
        end

        # Increase delta enough to advance the decoder's
        # <n,i> state to <m,0>, but guard against overflow:

        if m - n > (PUNYCODE_MAXINT - delta) / (h + 1)
          raise PunycodeOverflow, "Input needs wider integers to process."
        end
        delta += (m - n) * (h + 1)
        n = m

        input_length.times do |j|
          # Punycode does not need to check whether input[j] is basic:
          if input[j] < n
            delta += 1
            if delta == 0
              raise PunycodeOverflow,
                "Input needs wider integers to process."
            end
          end

          if input[j] == n
            # Represent delta as a generalized variable-length integer:

            q = delta; k = PUNYCODE_BASE
            while true
              if out >= max_out
                raise PunycodeBigOutput,
                  "Output would exceed the space provided."
              end
              t = (
                if k <= bias
                  PUNYCODE_TMIN
                elsif k >= bias + PUNYCODE_TMAX
                  PUNYCODE_TMAX
                else
                  k - bias
                end
              )
              break if q < t
              output[out] =
                punycode_encode_digit(t + (q - t) % (PUNYCODE_BASE - t))
              out += 1
              q = (q - t) / (PUNYCODE_BASE - t)
              k += PUNYCODE_BASE
            end

            output[out] = punycode_encode_digit(q)
            out += 1
            bias = punycode_adapt(delta, h + 1, h == b)
            delta = 0
            h += 1
          end
        end

        delta += 1
        n += 1
      end

      output_length[0] = out

      outlen = out
      outlen.times do |j|
        c = output[j]
        unless c >= 0 && c <= 127
          raise StandardError, "Invalid output char."
        end
        unless PUNYCODE_PRINT_ASCII[c]
          raise PunycodeBadInput, "Input is invalid."
        end
      end

      output[0..outlen].map { |x| x.chr }.join("").sub(/\0+\z/, "")
    end
    private_class_method :punycode_encode

    def self.punycode_decode(punycode)
      input = []
      output = []

      if ACE_MAX_LENGTH * 2 < punycode.size
        raise PunycodeBigOutput, "Output would exceed the space provided."
      end
      punycode.each_byte do |c|
        unless c >= 0 && c <= 127
          raise PunycodeBadInput, "Input is invalid."
        end
        input.push(c)
      end

      input_length = input.length
      output_length = [UNICODE_MAX_LENGTH]

      # Initialize the state
      n = PUNYCODE_INITIAL_N

      out = i = 0
      max_out = output_length[0]
      bias = PUNYCODE_INITIAL_BIAS

      # Handle the basic code points:  Let b be the number of input code
      # points before the last delimiter, or 0 if there is none, then
      # copy the first b code points to the output.

      b = 0
      input_length.times do |j|
        b = j if punycode_delimiter?(input[j])
      end
      if b > max_out
        raise PunycodeBigOutput, "Output would exceed the space provided."
      end

      b.times do |j|
        unless punycode_basic?(input[j])
          raise PunycodeBadInput, "Input is invalid."
        end
        output[out] = input[j]
        out+=1
      end

      # Main decoding loop:  Start just after the last delimiter if any
      # basic code points were copied; start at the beginning otherwise.

      in_ = b > 0 ? b + 1 : 0
      while in_ < input_length

        # in_ is the index of the next character to be consumed, and
        # out is the number of code points in the output array.

        # Decode a generalized variable-length integer into delta,
        # which gets added to i.  The overflow checking is easier
        # if we increase i as we go, then subtract off its starting
        # value at the end to obtain delta.

        oldi = i; w = 1; k = PUNYCODE_BASE
        while true
          if in_ >= input_length
            raise PunycodeBadInput, "Input is invalid."
          end
          digit = punycode_decode_digit(input[in_])
          in_+=1
          if digit >= PUNYCODE_BASE
            raise PunycodeBadInput, "Input is invalid."
          end
          if digit > (PUNYCODE_MAXINT - i) / w
            raise PunycodeOverflow, "Input needs wider integers to process."
          end
          i += digit * w
          t = (
            if k <= bias
              PUNYCODE_TMIN
            elsif k >= bias + PUNYCODE_TMAX
              PUNYCODE_TMAX
            else
              k - bias
            end
          )
          break if digit < t
          if w > PUNYCODE_MAXINT / (PUNYCODE_BASE - t)
            raise PunycodeOverflow, "Input needs wider integers to process."
          end
          w *= PUNYCODE_BASE - t
          k += PUNYCODE_BASE
        end

        bias = punycode_adapt(i - oldi, out + 1, oldi == 0)

        # I was supposed to wrap around from out + 1 to 0,
        # incrementing n each time, so we'll fix that now:

        if i / (out + 1) > PUNYCODE_MAXINT - n
          raise PunycodeOverflow, "Input needs wider integers to process."
        end
        n += i / (out + 1)
        i %= out + 1

        # Insert n at position i of the output:

        # not needed for Punycode:
        # raise PUNYCODE_INVALID_INPUT if decode_digit(n) <= base
        if out >= max_out
          raise PunycodeBigOutput, "Output would exceed the space provided."
        end

        #memmove(output + i + 1, output + i, (out - i) * sizeof *output)
        output[i + 1, out - i] = output[i, out - i]
        output[i] = n
        i += 1

        out += 1
      end

      output_length[0] = out

      output.pack("U*")
    end
    private_class_method :punycode_decode

    def self.punycode_basic?(codepoint)
      codepoint < 0x80
    end
    private_class_method :punycode_basic?

    def self.punycode_delimiter?(codepoint)
      codepoint == PUNYCODE_DELIMITER
    end
    private_class_method :punycode_delimiter?

    def self.punycode_encode_digit(d)
      d + 22 + 75 * ((d < 26) ? 1 : 0)
    end
    private_class_method :punycode_encode_digit

    # Returns the numeric value of a basic codepoint
    # (for use in representing integers) in the range 0 to
    # base - 1, or PUNYCODE_BASE if codepoint does not represent a value.
    def self.punycode_decode_digit(codepoint)
      if codepoint - 48 < 10
        codepoint - 22
      elsif codepoint - 65 < 26
        codepoint - 65
      elsif codepoint - 97 < 26
        codepoint - 97
      else
        PUNYCODE_BASE
      end
    end
    private_class_method :punycode_decode_digit

    # Bias adaptation method
    def self.punycode_adapt(delta, numpoints, firsttime)
      delta = firsttime ? delta / PUNYCODE_DAMP : delta >> 1
      # delta >> 1 is a faster way of doing delta / 2
      delta += delta / numpoints
      difference = PUNYCODE_BASE - PUNYCODE_TMIN

      k = 0
      while delta > (difference * PUNYCODE_TMAX) / 2
        delta /= difference
        k += PUNYCODE_BASE
      end

      k + (difference + 1) * delta / (delta + PUNYCODE_SKEW)
    end
    private_class_method :punycode_adapt
  end
  # :startdoc:
end
