# ----------------------------------------------------------------------------
#
#  Copyright (C) 2024 Fons Adriaensen <fons@linuxaudio.org>
#    
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http:#www.gnu.org/licenses/>.
#
# ----------------------------------------------------------------------------


from zita_jacktools.jackclient import JackClient
from zita_jacktools import jackfwcapt_ext


class JackFwcapt (JackClient):

    """
    JackFwcapt is used together with JackFwplay to process
    audio files via a Jack graph in freewheel mode. This
    class records the processed audio. Its operation is
    controlled via the '_sync' port by a JackFwplay client.
    """

    def __init__(self, nchan, client_name, server_name = None):
        """
        Create a new JackFwcapt instance.

        The result should be checked using get_state().
        The optional 'server_name' allows to select between running
        Jack servers.
        """
        assert (nchan > 0)
        self._jfwcapt, base = jackfwcapt_ext.makecaps (self, client_name, server_name, nchan)
        super().__init__(base)
        self.nchan = nchan
        

    def create_file (self, fname, fchan, ftype):
        """
        Create the audio file for writing. When the controlling
        JackFwplay stops sending data, this file is closed so it
        will be used just once.
        
        The ftype argument should be a string consisting of the
        words listed below, comma separated and without spaces.
        Dither is applied to 16-bit files only.

        Returns zero for success, non zero otherwise.
        
        File types:      caf, wav, amb, aiff, flac.
        Sample formats:  16bit, 24bit, 32bit, float. 
        Dither type:     none, rec, tri, lips.
        """
        return jackfwcapt_ext.create_file (self._jfwcapt, fname, fchan, ftype)


    def connect_sync (self, player):
        return self.connect_input (self.nchan, player + ':_sync')
 
