﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/textract/Textract_EXPORTS.h>
#include <aws/textract/model/AdapterVersionStatus.h>
#include <aws/textract/model/FeatureType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Textract {
namespace Model {

/**
 * <p>Summary info for an adapter version. Contains information on the AdapterId,
 * AdapterVersion, CreationTime, FeatureTypes, and Status.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/AdapterVersionOverview">AWS
 * API Reference</a></p>
 */
class AdapterVersionOverview {
 public:
  AWS_TEXTRACT_API AdapterVersionOverview() = default;
  AWS_TEXTRACT_API AdapterVersionOverview(Aws::Utils::Json::JsonView jsonValue);
  AWS_TEXTRACT_API AdapterVersionOverview& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_TEXTRACT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>A unique identifier for the adapter associated with a given adapter
   * version.</p>
   */
  inline const Aws::String& GetAdapterId() const { return m_adapterId; }
  inline bool AdapterIdHasBeenSet() const { return m_adapterIdHasBeenSet; }
  template <typename AdapterIdT = Aws::String>
  void SetAdapterId(AdapterIdT&& value) {
    m_adapterIdHasBeenSet = true;
    m_adapterId = std::forward<AdapterIdT>(value);
  }
  template <typename AdapterIdT = Aws::String>
  AdapterVersionOverview& WithAdapterId(AdapterIdT&& value) {
    SetAdapterId(std::forward<AdapterIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An identified for a given adapter version.</p>
   */
  inline const Aws::String& GetAdapterVersion() const { return m_adapterVersion; }
  inline bool AdapterVersionHasBeenSet() const { return m_adapterVersionHasBeenSet; }
  template <typename AdapterVersionT = Aws::String>
  void SetAdapterVersion(AdapterVersionT&& value) {
    m_adapterVersionHasBeenSet = true;
    m_adapterVersion = std::forward<AdapterVersionT>(value);
  }
  template <typename AdapterVersionT = Aws::String>
  AdapterVersionOverview& WithAdapterVersion(AdapterVersionT&& value) {
    SetAdapterVersion(std::forward<AdapterVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that a given adapter version was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  inline bool CreationTimeHasBeenSet() const { return m_creationTimeHasBeenSet; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  AdapterVersionOverview& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The feature types that the adapter version is operating on.</p>
   */
  inline const Aws::Vector<FeatureType>& GetFeatureTypes() const { return m_featureTypes; }
  inline bool FeatureTypesHasBeenSet() const { return m_featureTypesHasBeenSet; }
  template <typename FeatureTypesT = Aws::Vector<FeatureType>>
  void SetFeatureTypes(FeatureTypesT&& value) {
    m_featureTypesHasBeenSet = true;
    m_featureTypes = std::forward<FeatureTypesT>(value);
  }
  template <typename FeatureTypesT = Aws::Vector<FeatureType>>
  AdapterVersionOverview& WithFeatureTypes(FeatureTypesT&& value) {
    SetFeatureTypes(std::forward<FeatureTypesT>(value));
    return *this;
  }
  inline AdapterVersionOverview& AddFeatureTypes(FeatureType value) {
    m_featureTypesHasBeenSet = true;
    m_featureTypes.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains information on the status of a given adapter version.</p>
   */
  inline AdapterVersionStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(AdapterVersionStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline AdapterVersionOverview& WithStatus(AdapterVersionStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A message explaining the status of a given adapter vesion.</p>
   */
  inline const Aws::String& GetStatusMessage() const { return m_statusMessage; }
  inline bool StatusMessageHasBeenSet() const { return m_statusMessageHasBeenSet; }
  template <typename StatusMessageT = Aws::String>
  void SetStatusMessage(StatusMessageT&& value) {
    m_statusMessageHasBeenSet = true;
    m_statusMessage = std::forward<StatusMessageT>(value);
  }
  template <typename StatusMessageT = Aws::String>
  AdapterVersionOverview& WithStatusMessage(StatusMessageT&& value) {
    SetStatusMessage(std::forward<StatusMessageT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_adapterId;

  Aws::String m_adapterVersion;

  Aws::Utils::DateTime m_creationTime{};

  Aws::Vector<FeatureType> m_featureTypes;

  AdapterVersionStatus m_status{AdapterVersionStatus::NOT_SET};

  Aws::String m_statusMessage;
  bool m_adapterIdHasBeenSet = false;
  bool m_adapterVersionHasBeenSet = false;
  bool m_creationTimeHasBeenSet = false;
  bool m_featureTypesHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_statusMessageHasBeenSet = false;
};

}  // namespace Model
}  // namespace Textract
}  // namespace Aws
