﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotsitewise/IoTSiteWise_EXPORTS.h>
#include <aws/iotsitewise/model/PropertyDataType.h>
#include <aws/iotsitewise/model/PropertyType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace IoTSiteWise {
namespace Model {

/**
 * <p>Contains an asset model property definition. This property definition is
 * applied to all assets created from the asset model.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/iotsitewise-2019-12-02/AssetModelPropertyDefinition">AWS
 * API Reference</a></p>
 */
class AssetModelPropertyDefinition {
 public:
  AWS_IOTSITEWISE_API AssetModelPropertyDefinition() = default;
  AWS_IOTSITEWISE_API AssetModelPropertyDefinition(Aws::Utils::Json::JsonView jsonValue);
  AWS_IOTSITEWISE_API AssetModelPropertyDefinition& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_IOTSITEWISE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The ID to assign to the asset model property, if desired. IoT SiteWise
   * automatically generates a unique ID for you, so this parameter is never
   * required. However, if you prefer to supply your own ID instead, you can specify
   * it here in UUID format. If you specify your own ID, it must be globally
   * unique.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  AssetModelPropertyDefinition& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An external ID to assign to the property definition. The external ID must be
   * unique among property definitions within this asset model. For more information,
   * see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/object-ids.html#external-ids">Using
   * external IDs</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetExternalId() const { return m_externalId; }
  inline bool ExternalIdHasBeenSet() const { return m_externalIdHasBeenSet; }
  template <typename ExternalIdT = Aws::String>
  void SetExternalId(ExternalIdT&& value) {
    m_externalIdHasBeenSet = true;
    m_externalId = std::forward<ExternalIdT>(value);
  }
  template <typename ExternalIdT = Aws::String>
  AssetModelPropertyDefinition& WithExternalId(ExternalIdT&& value) {
    SetExternalId(std::forward<ExternalIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the property definition.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  AssetModelPropertyDefinition& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data type of the property definition.</p> <p>If you specify
   * <code>STRUCT</code>, you must also specify <code>dataTypeSpec</code> to identify
   * the type of the structure for this property.</p>
   */
  inline PropertyDataType GetDataType() const { return m_dataType; }
  inline bool DataTypeHasBeenSet() const { return m_dataTypeHasBeenSet; }
  inline void SetDataType(PropertyDataType value) {
    m_dataTypeHasBeenSet = true;
    m_dataType = value;
  }
  inline AssetModelPropertyDefinition& WithDataType(PropertyDataType value) {
    SetDataType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data type of the structure for this property. This parameter is required
   * on properties that have the <code>STRUCT</code> data type.</p> <p>The options
   * for this parameter depend on the type of the composite model in which you define
   * this property. Use <code>AWS/ALARM_STATE</code> for alarm state in alarm
   * composite models.</p>
   */
  inline const Aws::String& GetDataTypeSpec() const { return m_dataTypeSpec; }
  inline bool DataTypeSpecHasBeenSet() const { return m_dataTypeSpecHasBeenSet; }
  template <typename DataTypeSpecT = Aws::String>
  void SetDataTypeSpec(DataTypeSpecT&& value) {
    m_dataTypeSpecHasBeenSet = true;
    m_dataTypeSpec = std::forward<DataTypeSpecT>(value);
  }
  template <typename DataTypeSpecT = Aws::String>
  AssetModelPropertyDefinition& WithDataTypeSpec(DataTypeSpecT&& value) {
    SetDataTypeSpec(std::forward<DataTypeSpecT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unit of the property definition, such as <code>Newtons</code> or
   * <code>RPM</code>.</p>
   */
  inline const Aws::String& GetUnit() const { return m_unit; }
  inline bool UnitHasBeenSet() const { return m_unitHasBeenSet; }
  template <typename UnitT = Aws::String>
  void SetUnit(UnitT&& value) {
    m_unitHasBeenSet = true;
    m_unit = std::forward<UnitT>(value);
  }
  template <typename UnitT = Aws::String>
  AssetModelPropertyDefinition& WithUnit(UnitT&& value) {
    SetUnit(std::forward<UnitT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The property definition type (see <code>PropertyType</code>). You can only
   * specify one type in a property definition.</p>
   */
  inline const PropertyType& GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  template <typename TypeT = PropertyType>
  void SetType(TypeT&& value) {
    m_typeHasBeenSet = true;
    m_type = std::forward<TypeT>(value);
  }
  template <typename TypeT = PropertyType>
  AssetModelPropertyDefinition& WithType(TypeT&& value) {
    SetType(std::forward<TypeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  Aws::String m_externalId;

  Aws::String m_name;

  PropertyDataType m_dataType{PropertyDataType::NOT_SET};

  Aws::String m_dataTypeSpec;

  Aws::String m_unit;

  PropertyType m_type;
  bool m_idHasBeenSet = false;
  bool m_externalIdHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_dataTypeHasBeenSet = false;
  bool m_dataTypeSpecHasBeenSet = false;
  bool m_unitHasBeenSet = false;
  bool m_typeHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTSiteWise
}  // namespace Aws
