﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/MetadataProperties.h>
#include <aws/sagemaker/model/Tag.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class CreateTrialRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API CreateTrialRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTrial"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the trial. The name must be unique in your Amazon Web Services
   * account and is not case-sensitive.</p>
   */
  inline const Aws::String& GetTrialName() const { return m_trialName; }
  inline bool TrialNameHasBeenSet() const { return m_trialNameHasBeenSet; }
  template <typename TrialNameT = Aws::String>
  void SetTrialName(TrialNameT&& value) {
    m_trialNameHasBeenSet = true;
    m_trialName = std::forward<TrialNameT>(value);
  }
  template <typename TrialNameT = Aws::String>
  CreateTrialRequest& WithTrialName(TrialNameT&& value) {
    SetTrialName(std::forward<TrialNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the trial as displayed. The name doesn't need to be unique. If
   * <code>DisplayName</code> isn't specified, <code>TrialName</code> is
   * displayed.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  CreateTrialRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the experiment to associate the trial with.</p>
   */
  inline const Aws::String& GetExperimentName() const { return m_experimentName; }
  inline bool ExperimentNameHasBeenSet() const { return m_experimentNameHasBeenSet; }
  template <typename ExperimentNameT = Aws::String>
  void SetExperimentName(ExperimentNameT&& value) {
    m_experimentNameHasBeenSet = true;
    m_experimentName = std::forward<ExperimentNameT>(value);
  }
  template <typename ExperimentNameT = Aws::String>
  CreateTrialRequest& WithExperimentName(ExperimentNameT&& value) {
    SetExperimentName(std::forward<ExperimentNameT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const MetadataProperties& GetMetadataProperties() const { return m_metadataProperties; }
  inline bool MetadataPropertiesHasBeenSet() const { return m_metadataPropertiesHasBeenSet; }
  template <typename MetadataPropertiesT = MetadataProperties>
  void SetMetadataProperties(MetadataPropertiesT&& value) {
    m_metadataPropertiesHasBeenSet = true;
    m_metadataProperties = std::forward<MetadataPropertiesT>(value);
  }
  template <typename MetadataPropertiesT = MetadataProperties>
  CreateTrialRequest& WithMetadataProperties(MetadataPropertiesT&& value) {
    SetMetadataProperties(std::forward<MetadataPropertiesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags to associate with the trial. You can use <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_Search.html">Search</a>
   * API to search on the tags.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateTrialRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateTrialRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_trialName;

  Aws::String m_displayName;

  Aws::String m_experimentName;

  MetadataProperties m_metadataProperties;

  Aws::Vector<Tag> m_tags;
  bool m_trialNameHasBeenSet = false;
  bool m_displayNameHasBeenSet = false;
  bool m_experimentNameHasBeenSet = false;
  bool m_metadataPropertiesHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
