﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/pipes/Pipes_EXPORTS.h>
#include <aws/pipes/model/MQBrokerAccessCredentials.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Pipes {
namespace Model {

/**
 * <p>The parameters for using a Rabbit MQ broker as a source.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/PipeSourceRabbitMQBrokerParameters">AWS
 * API Reference</a></p>
 */
class PipeSourceRabbitMQBrokerParameters {
 public:
  AWS_PIPES_API PipeSourceRabbitMQBrokerParameters() = default;
  AWS_PIPES_API PipeSourceRabbitMQBrokerParameters(Aws::Utils::Json::JsonView jsonValue);
  AWS_PIPES_API PipeSourceRabbitMQBrokerParameters& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PIPES_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The credentials needed to access the resource.</p>
   */
  inline const MQBrokerAccessCredentials& GetCredentials() const { return m_credentials; }
  inline bool CredentialsHasBeenSet() const { return m_credentialsHasBeenSet; }
  template <typename CredentialsT = MQBrokerAccessCredentials>
  void SetCredentials(CredentialsT&& value) {
    m_credentialsHasBeenSet = true;
    m_credentials = std::forward<CredentialsT>(value);
  }
  template <typename CredentialsT = MQBrokerAccessCredentials>
  PipeSourceRabbitMQBrokerParameters& WithCredentials(CredentialsT&& value) {
    SetCredentials(std::forward<CredentialsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the destination queue to consume.</p>
   */
  inline const Aws::String& GetQueueName() const { return m_queueName; }
  inline bool QueueNameHasBeenSet() const { return m_queueNameHasBeenSet; }
  template <typename QueueNameT = Aws::String>
  void SetQueueName(QueueNameT&& value) {
    m_queueNameHasBeenSet = true;
    m_queueName = std::forward<QueueNameT>(value);
  }
  template <typename QueueNameT = Aws::String>
  PipeSourceRabbitMQBrokerParameters& WithQueueName(QueueNameT&& value) {
    SetQueueName(std::forward<QueueNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the virtual host associated with the source broker.</p>
   */
  inline const Aws::String& GetVirtualHost() const { return m_virtualHost; }
  inline bool VirtualHostHasBeenSet() const { return m_virtualHostHasBeenSet; }
  template <typename VirtualHostT = Aws::String>
  void SetVirtualHost(VirtualHostT&& value) {
    m_virtualHostHasBeenSet = true;
    m_virtualHost = std::forward<VirtualHostT>(value);
  }
  template <typename VirtualHostT = Aws::String>
  PipeSourceRabbitMQBrokerParameters& WithVirtualHost(VirtualHostT&& value) {
    SetVirtualHost(std::forward<VirtualHostT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in each batch.</p>
   */
  inline int GetBatchSize() const { return m_batchSize; }
  inline bool BatchSizeHasBeenSet() const { return m_batchSizeHasBeenSet; }
  inline void SetBatchSize(int value) {
    m_batchSizeHasBeenSet = true;
    m_batchSize = value;
  }
  inline PipeSourceRabbitMQBrokerParameters& WithBatchSize(int value) {
    SetBatchSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum length of a time to wait for events.</p>
   */
  inline int GetMaximumBatchingWindowInSeconds() const { return m_maximumBatchingWindowInSeconds; }
  inline bool MaximumBatchingWindowInSecondsHasBeenSet() const { return m_maximumBatchingWindowInSecondsHasBeenSet; }
  inline void SetMaximumBatchingWindowInSeconds(int value) {
    m_maximumBatchingWindowInSecondsHasBeenSet = true;
    m_maximumBatchingWindowInSeconds = value;
  }
  inline PipeSourceRabbitMQBrokerParameters& WithMaximumBatchingWindowInSeconds(int value) {
    SetMaximumBatchingWindowInSeconds(value);
    return *this;
  }
  ///@}
 private:
  MQBrokerAccessCredentials m_credentials;

  Aws::String m_queueName;

  Aws::String m_virtualHost;

  int m_batchSize{0};

  int m_maximumBatchingWindowInSeconds{0};
  bool m_credentialsHasBeenSet = false;
  bool m_queueNameHasBeenSet = false;
  bool m_virtualHostHasBeenSet = false;
  bool m_batchSizeHasBeenSet = false;
  bool m_maximumBatchingWindowInSecondsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Pipes
}  // namespace Aws
