﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mq/MQRequest.h>
#include <aws/mq/MQ_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace MQ {
namespace Model {

/**
 */
class ListUsersRequest : public MQRequest {
 public:
  AWS_MQ_API ListUsersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListUsers"; }

  AWS_MQ_API Aws::String SerializePayload() const override;

  AWS_MQ_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The unique ID that Amazon MQ generates for the broker.</p>
   */
  inline const Aws::String& GetBrokerId() const { return m_brokerId; }
  inline bool BrokerIdHasBeenSet() const { return m_brokerIdHasBeenSet; }
  template <typename BrokerIdT = Aws::String>
  void SetBrokerId(BrokerIdT&& value) {
    m_brokerIdHasBeenSet = true;
    m_brokerId = std::forward<BrokerIdT>(value);
  }
  template <typename BrokerIdT = Aws::String>
  ListUsersRequest& WithBrokerId(BrokerIdT&& value) {
    SetBrokerId(std::forward<BrokerIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of brokers that Amazon MQ can return per page (20 by
   * default). This value must be an integer from 5 to 100.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListUsersRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token that specifies the next page of results Amazon MQ should return. To
   * request the first page, leave nextToken empty.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListUsersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_brokerId;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_brokerIdHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace MQ
}  // namespace Aws
