﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mediapackage-vod/MediaPackageVodRequest.h>
#include <aws/mediapackage-vod/MediaPackageVod_EXPORTS.h>
#include <aws/mediapackage-vod/model/Authorization.h>
#include <aws/mediapackage-vod/model/EgressAccessLogs.h>

#include <utility>

namespace Aws {
namespace MediaPackageVod {
namespace Model {

/**
 * A new MediaPackage VOD PackagingGroup resource configuration.<p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreatePackagingGroupRequest">AWS
 * API Reference</a></p>
 */
class CreatePackagingGroupRequest : public MediaPackageVodRequest {
 public:
  AWS_MEDIAPACKAGEVOD_API CreatePackagingGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreatePackagingGroup"; }

  AWS_MEDIAPACKAGEVOD_API Aws::String SerializePayload() const override;

  ///@{

  inline const Authorization& GetAuthorization() const { return m_authorization; }
  inline bool AuthorizationHasBeenSet() const { return m_authorizationHasBeenSet; }
  template <typename AuthorizationT = Authorization>
  void SetAuthorization(AuthorizationT&& value) {
    m_authorizationHasBeenSet = true;
    m_authorization = std::forward<AuthorizationT>(value);
  }
  template <typename AuthorizationT = Authorization>
  CreatePackagingGroupRequest& WithAuthorization(AuthorizationT&& value) {
    SetAuthorization(std::forward<AuthorizationT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const EgressAccessLogs& GetEgressAccessLogs() const { return m_egressAccessLogs; }
  inline bool EgressAccessLogsHasBeenSet() const { return m_egressAccessLogsHasBeenSet; }
  template <typename EgressAccessLogsT = EgressAccessLogs>
  void SetEgressAccessLogs(EgressAccessLogsT&& value) {
    m_egressAccessLogsHasBeenSet = true;
    m_egressAccessLogs = std::forward<EgressAccessLogsT>(value);
  }
  template <typename EgressAccessLogsT = EgressAccessLogs>
  CreatePackagingGroupRequest& WithEgressAccessLogs(EgressAccessLogsT&& value) {
    SetEgressAccessLogs(std::forward<EgressAccessLogsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The ID of the PackagingGroup.
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  CreatePackagingGroupRequest& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreatePackagingGroupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreatePackagingGroupRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Authorization m_authorization;

  EgressAccessLogs m_egressAccessLogs;

  Aws::String m_id;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_authorizationHasBeenSet = false;
  bool m_egressAccessLogsHasBeenSet = false;
  bool m_idHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaPackageVod
}  // namespace Aws
