﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/frauddetector/FraudDetectorRequest.h>
#include <aws/frauddetector/FraudDetector_EXPORTS.h>
#include <aws/frauddetector/model/FilterCondition.h>
#include <aws/frauddetector/model/PredictionTimeRange.h>

#include <utility>

namespace Aws {
namespace FraudDetector {
namespace Model {

/**
 */
class ListEventPredictionsRequest : public FraudDetectorRequest {
 public:
  AWS_FRAUDDETECTOR_API ListEventPredictionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListEventPredictions"; }

  AWS_FRAUDDETECTOR_API Aws::String SerializePayload() const override;

  AWS_FRAUDDETECTOR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p> The event ID. </p>
   */
  inline const FilterCondition& GetEventId() const { return m_eventId; }
  inline bool EventIdHasBeenSet() const { return m_eventIdHasBeenSet; }
  template <typename EventIdT = FilterCondition>
  void SetEventId(EventIdT&& value) {
    m_eventIdHasBeenSet = true;
    m_eventId = std::forward<EventIdT>(value);
  }
  template <typename EventIdT = FilterCondition>
  ListEventPredictionsRequest& WithEventId(EventIdT&& value) {
    SetEventId(std::forward<EventIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The event type associated with the detector. </p>
   */
  inline const FilterCondition& GetEventType() const { return m_eventType; }
  inline bool EventTypeHasBeenSet() const { return m_eventTypeHasBeenSet; }
  template <typename EventTypeT = FilterCondition>
  void SetEventType(EventTypeT&& value) {
    m_eventTypeHasBeenSet = true;
    m_eventType = std::forward<EventTypeT>(value);
  }
  template <typename EventTypeT = FilterCondition>
  ListEventPredictionsRequest& WithEventType(EventTypeT&& value) {
    SetEventType(std::forward<EventTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The detector ID. </p>
   */
  inline const FilterCondition& GetDetectorId() const { return m_detectorId; }
  inline bool DetectorIdHasBeenSet() const { return m_detectorIdHasBeenSet; }
  template <typename DetectorIdT = FilterCondition>
  void SetDetectorId(DetectorIdT&& value) {
    m_detectorIdHasBeenSet = true;
    m_detectorId = std::forward<DetectorIdT>(value);
  }
  template <typename DetectorIdT = FilterCondition>
  ListEventPredictionsRequest& WithDetectorId(DetectorIdT&& value) {
    SetDetectorId(std::forward<DetectorIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The detector version ID. </p>
   */
  inline const FilterCondition& GetDetectorVersionId() const { return m_detectorVersionId; }
  inline bool DetectorVersionIdHasBeenSet() const { return m_detectorVersionIdHasBeenSet; }
  template <typename DetectorVersionIdT = FilterCondition>
  void SetDetectorVersionId(DetectorVersionIdT&& value) {
    m_detectorVersionIdHasBeenSet = true;
    m_detectorVersionId = std::forward<DetectorVersionIdT>(value);
  }
  template <typename DetectorVersionIdT = FilterCondition>
  ListEventPredictionsRequest& WithDetectorVersionId(DetectorVersionIdT&& value) {
    SetDetectorVersionId(std::forward<DetectorVersionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The time period for when the predictions were generated. </p>
   */
  inline const PredictionTimeRange& GetPredictionTimeRange() const { return m_predictionTimeRange; }
  inline bool PredictionTimeRangeHasBeenSet() const { return m_predictionTimeRangeHasBeenSet; }
  template <typename PredictionTimeRangeT = PredictionTimeRange>
  void SetPredictionTimeRange(PredictionTimeRangeT&& value) {
    m_predictionTimeRangeHasBeenSet = true;
    m_predictionTimeRange = std::forward<PredictionTimeRangeT>(value);
  }
  template <typename PredictionTimeRangeT = PredictionTimeRange>
  ListEventPredictionsRequest& WithPredictionTimeRange(PredictionTimeRangeT&& value) {
    SetPredictionTimeRange(std::forward<PredictionTimeRangeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Identifies the next page of results to return. Use the token to make the
   * call again to retrieve the next page. Keep all other arguments unchanged. Each
   * pagination token expires after 24 hours. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListEventPredictionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The maximum number of predictions to return for the request. </p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListEventPredictionsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  FilterCondition m_eventId;

  FilterCondition m_eventType;

  FilterCondition m_detectorId;

  FilterCondition m_detectorVersionId;

  PredictionTimeRange m_predictionTimeRange;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_eventIdHasBeenSet = false;
  bool m_eventTypeHasBeenSet = false;
  bool m_detectorIdHasBeenSet = false;
  bool m_detectorVersionIdHasBeenSet = false;
  bool m_predictionTimeRangeHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace FraudDetector
}  // namespace Aws
