﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/compute-optimizer/ComputeOptimizerRequest.h>
#include <aws/compute-optimizer/ComputeOptimizer_EXPORTS.h>
#include <aws/compute-optimizer/model/ECSServiceRecommendationFilter.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace ComputeOptimizer {
namespace Model {

/**
 */
class GetECSServiceRecommendationsRequest : public ComputeOptimizerRequest {
 public:
  AWS_COMPUTEOPTIMIZER_API GetECSServiceRecommendationsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetECSServiceRecommendations"; }

  AWS_COMPUTEOPTIMIZER_API Aws::String SerializePayload() const override;

  AWS_COMPUTEOPTIMIZER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p> The ARN that identifies the Amazon ECS service. </p> <p> The following is
   * the format of the ARN: </p> <p>
   * <code>arn:aws:ecs:region:aws_account_id:service/cluster-name/service-name</code>
   * </p>
   */
  inline const Aws::Vector<Aws::String>& GetServiceArns() const { return m_serviceArns; }
  inline bool ServiceArnsHasBeenSet() const { return m_serviceArnsHasBeenSet; }
  template <typename ServiceArnsT = Aws::Vector<Aws::String>>
  void SetServiceArns(ServiceArnsT&& value) {
    m_serviceArnsHasBeenSet = true;
    m_serviceArns = std::forward<ServiceArnsT>(value);
  }
  template <typename ServiceArnsT = Aws::Vector<Aws::String>>
  GetECSServiceRecommendationsRequest& WithServiceArns(ServiceArnsT&& value) {
    SetServiceArns(std::forward<ServiceArnsT>(value));
    return *this;
  }
  template <typename ServiceArnsT = Aws::String>
  GetECSServiceRecommendationsRequest& AddServiceArns(ServiceArnsT&& value) {
    m_serviceArnsHasBeenSet = true;
    m_serviceArns.emplace_back(std::forward<ServiceArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The token to advance to the next page of Amazon ECS service recommendations.
   * </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  GetECSServiceRecommendationsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The maximum number of Amazon ECS service recommendations to return with a
   * single request. </p> <p>To retrieve the remaining results, make another request
   * with the returned <code>nextToken</code> value.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline GetECSServiceRecommendationsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> An array of objects to specify a filter that returns a more specific list of
   * Amazon ECS service recommendations. </p>
   */
  inline const Aws::Vector<ECSServiceRecommendationFilter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<ECSServiceRecommendationFilter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<ECSServiceRecommendationFilter>>
  GetECSServiceRecommendationsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = ECSServiceRecommendationFilter>
  GetECSServiceRecommendationsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Return the Amazon ECS service recommendations to the specified Amazon Web
   * Services account IDs. </p> <p>If your account is the management account or the
   * delegated administrator of an organization, use this parameter to return the
   * Amazon ECS service recommendations to specific member accounts.</p> <p>You can
   * only specify one account ID per request.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAccountIds() const { return m_accountIds; }
  inline bool AccountIdsHasBeenSet() const { return m_accountIdsHasBeenSet; }
  template <typename AccountIdsT = Aws::Vector<Aws::String>>
  void SetAccountIds(AccountIdsT&& value) {
    m_accountIdsHasBeenSet = true;
    m_accountIds = std::forward<AccountIdsT>(value);
  }
  template <typename AccountIdsT = Aws::Vector<Aws::String>>
  GetECSServiceRecommendationsRequest& WithAccountIds(AccountIdsT&& value) {
    SetAccountIds(std::forward<AccountIdsT>(value));
    return *this;
  }
  template <typename AccountIdsT = Aws::String>
  GetECSServiceRecommendationsRequest& AddAccountIds(AccountIdsT&& value) {
    m_accountIdsHasBeenSet = true;
    m_accountIds.emplace_back(std::forward<AccountIdsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_serviceArns;

  Aws::String m_nextToken;

  int m_maxResults{0};

  Aws::Vector<ECSServiceRecommendationFilter> m_filters;

  Aws::Vector<Aws::String> m_accountIds;
  bool m_serviceArnsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_accountIdsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ComputeOptimizer
}  // namespace Aws
