﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/application-insights/ApplicationInsightsRequest.h>
#include <aws/application-insights/ApplicationInsights_EXPORTS.h>
#include <aws/application-insights/model/Visibility.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ApplicationInsights {
namespace Model {

/**
 */
class ListProblemsRequest : public ApplicationInsightsRequest {
 public:
  AWS_APPLICATIONINSIGHTS_API ListProblemsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListProblems"; }

  AWS_APPLICATIONINSIGHTS_API Aws::String SerializePayload() const override;

  AWS_APPLICATIONINSIGHTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Web Services account ID for the resource group owner.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  ListProblemsRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the resource group.</p>
   */
  inline const Aws::String& GetResourceGroupName() const { return m_resourceGroupName; }
  inline bool ResourceGroupNameHasBeenSet() const { return m_resourceGroupNameHasBeenSet; }
  template <typename ResourceGroupNameT = Aws::String>
  void SetResourceGroupName(ResourceGroupNameT&& value) {
    m_resourceGroupNameHasBeenSet = true;
    m_resourceGroupName = std::forward<ResourceGroupNameT>(value);
  }
  template <typename ResourceGroupNameT = Aws::String>
  ListProblemsRequest& WithResourceGroupName(ResourceGroupNameT&& value) {
    SetResourceGroupName(std::forward<ResourceGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time when the problem was detected, in epoch seconds. If you don't
   * specify a time frame for the request, problems within the past seven days are
   * returned.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ListProblemsRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time when the problem ended, in epoch seconds. If not specified, problems
   * within the past seven days are returned.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ListProblemsRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return in a single call. To retrieve the
   * remaining results, make another call with the returned <code>NextToken</code>
   * value.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListProblemsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to request the next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListProblemsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The name of the component. </p>
   */
  inline const Aws::String& GetComponentName() const { return m_componentName; }
  inline bool ComponentNameHasBeenSet() const { return m_componentNameHasBeenSet; }
  template <typename ComponentNameT = Aws::String>
  void SetComponentName(ComponentNameT&& value) {
    m_componentNameHasBeenSet = true;
    m_componentName = std::forward<ComponentNameT>(value);
  }
  template <typename ComponentNameT = Aws::String>
  ListProblemsRequest& WithComponentName(ComponentNameT&& value) {
    SetComponentName(std::forward<ComponentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether or not you can view the problem. If not specified, visible
   * and ignored problems are returned.</p>
   */
  inline Visibility GetVisibility() const { return m_visibility; }
  inline bool VisibilityHasBeenSet() const { return m_visibilityHasBeenSet; }
  inline void SetVisibility(Visibility value) {
    m_visibilityHasBeenSet = true;
    m_visibility = value;
  }
  inline ListProblemsRequest& WithVisibility(Visibility value) {
    SetVisibility(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_accountId;

  Aws::String m_resourceGroupName;

  Aws::Utils::DateTime m_startTime{};

  Aws::Utils::DateTime m_endTime{};

  int m_maxResults{0};

  Aws::String m_nextToken;

  Aws::String m_componentName;

  Visibility m_visibility{Visibility::NOT_SET};
  bool m_accountIdHasBeenSet = false;
  bool m_resourceGroupNameHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_componentNameHasBeenSet = false;
  bool m_visibilityHasBeenSet = false;
};

}  // namespace Model
}  // namespace ApplicationInsights
}  // namespace Aws
