﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/wafv2/WAFV2Request.h>
#include <aws/wafv2/WAFV2_EXPORTS.h>
#include <aws/wafv2/model/Regex.h>
#include <aws/wafv2/model/Scope.h>
#include <aws/wafv2/model/Tag.h>

#include <utility>

namespace Aws {
namespace WAFV2 {
namespace Model {

/**
 */
class CreateRegexPatternSetRequest : public WAFV2Request {
 public:
  AWS_WAFV2_API CreateRegexPatternSetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRegexPatternSet"; }

  AWS_WAFV2_API Aws::String SerializePayload() const override;

  AWS_WAFV2_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the set. You cannot change the name after you create the set.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateRegexPatternSetRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether this is for a global resource type, such as a Amazon
   * CloudFront distribution. For an Amplify application, use
   * <code>CLOUDFRONT</code>.</p> <p>To work with CloudFront, you must also specify
   * the Region US East (N. Virginia) as follows: </p> <ul> <li> <p>CLI - Specify the
   * Region when you use the CloudFront scope: <code>--scope=CLOUDFRONT
   * --region=us-east-1</code>. </p> </li> <li> <p>API and SDKs - For all calls, use
   * the Region endpoint us-east-1. </p> </li> </ul>
   */
  inline Scope GetScope() const { return m_scope; }
  inline bool ScopeHasBeenSet() const { return m_scopeHasBeenSet; }
  inline void SetScope(Scope value) {
    m_scopeHasBeenSet = true;
    m_scope = value;
  }
  inline CreateRegexPatternSetRequest& WithScope(Scope value) {
    SetScope(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the set that helps with identification. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateRegexPatternSetRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Array of regular expression strings. </p>
   */
  inline const Aws::Vector<Regex>& GetRegularExpressionList() const { return m_regularExpressionList; }
  inline bool RegularExpressionListHasBeenSet() const { return m_regularExpressionListHasBeenSet; }
  template <typename RegularExpressionListT = Aws::Vector<Regex>>
  void SetRegularExpressionList(RegularExpressionListT&& value) {
    m_regularExpressionListHasBeenSet = true;
    m_regularExpressionList = std::forward<RegularExpressionListT>(value);
  }
  template <typename RegularExpressionListT = Aws::Vector<Regex>>
  CreateRegexPatternSetRequest& WithRegularExpressionList(RegularExpressionListT&& value) {
    SetRegularExpressionList(std::forward<RegularExpressionListT>(value));
    return *this;
  }
  template <typename RegularExpressionListT = Regex>
  CreateRegexPatternSetRequest& AddRegularExpressionList(RegularExpressionListT&& value) {
    m_regularExpressionListHasBeenSet = true;
    m_regularExpressionList.emplace_back(std::forward<RegularExpressionListT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of key:value pairs to associate with the resource.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateRegexPatternSetRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateRegexPatternSetRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Scope m_scope{Scope::NOT_SET};
  bool m_scopeHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::Vector<Regex> m_regularExpressionList;
  bool m_regularExpressionListHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace WAFV2
}  // namespace Aws
